<?php
/**
 * Obtiene los datos de la tabla de inscripción de cursos
 * Retorna todos los registros en formato JSON
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$nombreTabla = $data['nombreTabla'] ?? '';
$securityKey = $data['securityKey'] ?? '';

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logConsulta($message) {
    $logFile = '../logs/consulta_datos_tabla_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de consulta
    logConsulta("Consulta de datos - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Usuario: $dbUser");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Verificar si la tabla existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logConsulta("Error: Tabla no existe - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'La tabla no existe en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Obtener total de registros
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM `$nombreTabla`");
    $stmt->execute();
    $total = $stmt->fetch()['total'];
    
    // Obtener todos los datos de la tabla
    $stmt = $pdo->prepare("SELECT * FROM `$nombreTabla` ORDER BY id ASC LIMIT 1000"); // Limitar a 1000 registros por seguridad
    $stmt->execute();
    $datos = $stmt->fetchAll();
    
    logConsulta("Consulta exitosa - Total registros: $total, Registros retornados: " . count($datos));
    
    // Retornar datos
    echo json_encode([
        'success' => true,
        'total' => $total,
        'datos' => $datos,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de conexión
    logConsulta("Error de conexión: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error de conexión a la base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    logConsulta("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?> 