<?php
/**
 * Inicializa la tabla de inscripción de cursos agregando nuevos registros
 * con IDs incrementales y todos los campos de cursos en NULL
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$nombreTabla = $data['nombreTabla'] ?? '';
$securityKey = $data['securityKey'] ?? '';
$cantidadIds = intval($data['cantidadIds'] ?? 0);
$idInicio = intval($data['idInicio'] ?? 0);

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar parámetros de inicialización
if ($cantidadIds <= 0 || $cantidadIds > 1000) {
    http_response_code(400);
    echo json_encode(['error' => 'Cantidad de IDs debe estar entre 1 y 1000']);
    exit;
}

if ($idInicio <= 0) {
    http_response_code(400);
    echo json_encode(['error' => 'ID de inicio debe ser mayor a 0']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logInicializacion($message) {
    $logFile = '../logs/inicializacion_tabla_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de inicialización
    logInicializacion("Inicialización de tabla - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Cantidad: $cantidadIds, ID Inicio: $idInicio");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Verificar si la tabla existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logInicializacion("Error: Tabla no existe - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'La tabla no existe en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Verificar si ya existen algunos de los IDs que se quieren insertar
    $idFin = $idInicio + $cantidadIds - 1;
    $stmt = $pdo->prepare("SELECT COUNT(*) as existentes FROM `$nombreTabla` WHERE id BETWEEN ? AND ?");
    $stmt->execute([$idInicio, $idFin]);
    $existentes = $stmt->fetch()['existentes'];
    
    if ($existentes > 0) {
        logInicializacion("Advertencia: Ya existen $existentes IDs en el rango $idInicio-$idFin");
        
        echo json_encode([
            'success' => false,
            'message' => "Ya existen $existentes IDs en el rango $idInicio-$idFin. Por favor, elige otro rango de IDs.",
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Comenzar transacción
    $pdo->beginTransaction();
    
    // Preparar la consulta de inserción
    $sql = "INSERT INTO `$nombreTabla` (id, c001, c002, c003, c004, c005, c006, c007, c008, c009, c010) VALUES (?, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL)";
    $stmt = $pdo->prepare($sql);
    
    $registrosInsertados = 0;
    $idsInsertados = [];
    
    // Insertar registros uno por uno
    for ($i = 0; $i < $cantidadIds; $i++) {
        $idActual = $idInicio + $i;
        
        $stmt->execute([$idActual]);
        $registrosInsertados++;
        $idsInsertados[] = $idActual;
    }
    
    // Confirmar transacción
    $pdo->commit();
    
    logInicializacion("Inicialización exitosa - Registros insertados: $registrosInsertados, IDs: " . implode(', ', $idsInsertados));
    
    // Retornar resultado exitoso
    echo json_encode([
        'success' => true,
        'message' => "Tabla inicializada exitosamente con $registrosInsertados nuevos registros",
        'registrosInsertados' => $registrosInsertados,
        'idInicio' => $idInicio,
        'idFin' => $idFin,
        'idsInsertados' => $idsInsertados,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de base de datos
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    logInicializacion("Error de base de datos: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error de base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    logInicializacion("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?> 