<?php
/**
 * Ejecutor de script SQL para crear tabla de inscripción de cursos
 * Ejecuta el script SQL directamente en la base de datos del servidor
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$sqlContent = $data['sqlContent'] ?? '';
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$securityKey = $data['securityKey'] ?? '';

// Validar datos requeridos
if (empty($sqlContent) || empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Validar que el SQL contenga solo comandos CREATE TABLE
$sqlContent = trim($sqlContent);
if (!preg_match('/^CREATE\s+TABLE/i', $sqlContent)) {
    http_response_code(400);
    echo json_encode(['error' => 'El script SQL debe contener solo comandos CREATE TABLE']);
    exit;
}

// Función para registrar logs
function logExecution($message) {
    $logFile = '../logs/tabla_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de ejecución
    logExecution("Intento de creación de tabla - Host: $dbHost, DB: $dbName, Usuario: $dbUser");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Dividir el script SQL en comandos individuales
    $commands = array_filter(array_map('trim', explode(';', $sqlContent)));
    
    $results = [];
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($commands as $command) {
        if (empty($command)) continue;
        
        try {
            // Ejecutar comando
            $stmt = $pdo->prepare($command);
            $stmt->execute();
            
            $results[] = [
                'command' => substr($command, 0, 100) . '...',
                'status' => 'success',
                'message' => 'Comando ejecutado correctamente'
            ];
            $successCount++;
            
        } catch (PDOException $e) {
            $results[] = [
                'command' => substr($command, 0, 100) . '...',
                'status' => 'error',
                'message' => $e->getMessage()
            ];
            $errorCount++;
        }
    }
    
    // Cerrar conexión
    $pdo = null;
    
    // Registrar resultado
    logExecution("Ejecución completada - Éxitos: $successCount, Errores: $errorCount");
    
    // Preparar respuesta
    $response = [
        'success' => true,
        'message' => "Script ejecutado correctamente",
        'summary' => [
            'total_commands' => count($commands),
            'success_count' => $successCount,
            'error_count' => $errorCount
        ],
        'details' => $results,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    // Si hay errores, cambiar el estado
    if ($errorCount > 0) {
        $response['success'] = false;
        $response['message'] = "Script ejecutado con errores";
    }
    
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
} catch (PDOException $e) {
    // Error de conexión
    logExecution("Error de conexión: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Error de conexión a la base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    logExecution("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?> 