<?php
session_start();
header('Content-Type: application/json');

// Verificar que el usuario esté logueado
if (!isset($_SESSION['id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Usuario no autenticado']);
    exit();
}

require_once '../conexion.php';

try {
    // Obtener datos del POST
    $input = json_decode(file_get_contents('php://input'), true);
    
    $curso = $input['curso'] ?? '';
    $unidad = $input['unidad'] ?? '';
    $variable = $input['variable'] ?? '';
    $usuario_id = $input['usuario_id'] ?? $_SESSION['id'];
    
    // Validar parámetros
    if (empty($curso) || empty($unidad) || empty($variable)) {
        throw new Exception('Faltan parámetros requeridos');
    }
    
    // Validar que el usuario solo pueda resetear sus propias variables
    if ($usuario_id != $_SESSION['id']) {
        throw new Exception('No autorizado para resetear variables de otro usuario');
    }
    
    // Normalizar formato de unidad
    if (preg_match('/Unidad (\d+)/', $unidad, $matches)) {
        // Si viene como "Unidad 01"
        $unidad = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
    } else {
        // Si ya viene como "01" o similar
        $unidad = str_pad($unidad, 2, '0', STR_PAD_LEFT);
    }
    
    // Nombre de la tabla
    $nombre_tabla = "avance_{$curso}_u{$unidad}";
    
    // Obtener el tipo y valor por defecto de la columna
    $sql_info = "SELECT COLUMN_TYPE, COLUMN_DEFAULT 
                 FROM INFORMATION_SCHEMA.COLUMNS 
                 WHERE TABLE_SCHEMA = DATABASE() 
                 AND TABLE_NAME = :tabla 
                 AND COLUMN_NAME = :variable";
    $stmt_info = $dsn->prepare($sql_info);
    $stmt_info->bindParam(':tabla', $nombre_tabla, PDO::PARAM_STR);
    $stmt_info->bindParam(':variable', $variable, PDO::PARAM_STR);
    $stmt_info->execute();
    $info_columna = $stmt_info->fetch(PDO::FETCH_ASSOC);
    
    if (!$info_columna) {
        throw new Exception('Variable no encontrada');
    }
    
    // Determinar el valor de reinicio
    $valor_reset = $info_columna['COLUMN_DEFAULT'];
    if ($valor_reset === null) {
        // Si no hay valor por defecto, usar NULL
        $valor_reset = null;
    } elseif (strpos($info_columna['COLUMN_TYPE'], 'int') !== false) {
        // Para enteros, usar 0
        $valor_reset = 0;
    } else {
        // Para otros tipos, usar NULL
        $valor_reset = null;
    }
    
    // Actualizar la variable
    $sql_update = "UPDATE `{$nombre_tabla}` SET `{$variable}` = :valor WHERE id = :usuario_id";
    $stmt_update = $dsn->prepare($sql_update);
    
    if ($valor_reset === null) {
        $stmt_update->bindValue(':valor', null, PDO::PARAM_NULL);
    } else {
        $stmt_update->bindValue(':valor', $valor_reset, PDO::PARAM_INT);
    }
    
    $stmt_update->bindParam(':usuario_id', $usuario_id, PDO::PARAM_INT);
    $stmt_update->execute();
    
    echo json_encode([
        'success' => true,
        'message' => "Variable '{$variable}' reiniciada exitosamente",
        'valor_anterior' => null,
        'valor_nuevo' => $valor_reset
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    error_log("Error en reiniciar_variable.php: " . $e->getMessage());
}
?>
