<?php
session_start();

// Incluir configuración de cursos (desde la carpeta padre)
require_once '../configuracion_cursos.php';

// Obtener parámetros de la URL
$curso = isset($_GET['curso']) ? $_GET['curso'] : '';
$unidad = isset($_GET['unidad']) ? $_GET['unidad'] : '';
$actividad = isset($_GET['actividad']) ? $_GET['actividad'] : '';
$modo = isset($_GET['modo']) ? $_GET['modo'] : 'crear'; // 'crear' o 'editar'

// Generar nombre del archivo de actividad
$nombre_archivo = '';
$archivo_existe = false;
$contenido_archivo = '';

if ($curso && $unidad && $actividad) {
    // Extraer número de unidad del formato "Unidad X"
    $numero_unidad = '';
    if (preg_match('/Unidad (\d+)/', $unidad, $matches)) {
        $numero_unidad = sprintf('%02d', $matches[1]); // Formato con 2 dígitos: 01, 02, etc.
    }
    
    // Generar nombre del archivo: cXXX_uXX_actXX.php
    $numero_actividad = sprintf('%02d', $actividad); // Formato con 2 dígitos
    $nombre_archivo = "{$curso}_u{$numero_unidad}_act{$numero_actividad}.php";
    
    // Verificar si el archivo existe (en la carpeta recursos)
    $ruta_archivo = "../recursos/{$nombre_archivo}";
    $archivo_existe = file_exists($ruta_archivo);
    
// Si existe, leer el contenido y detectar secciones
$secciones_detectadas = [];
    if ($archivo_existe) {
        $contenido_archivo = file_get_contents($ruta_archivo);
    $secciones_detectadas = detectarSeccionesEnArchivo($contenido_archivo);
    }
}

?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configuración de Actividad - <?php echo $modo === 'crear' ? 'Crear' : 'Editar'; ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet"/>
    <link rel="stylesheet" href="../general.css" />
    <!-- TinyMCE -->
    <script src="https://cdn.tiny.cloud/1/no-api-key/tinymce/6/tinymce.min.js" referrerpolicy="origin"></script>
    <style>
        body {
            font-family: "Nunito", sans-serif;
            background: radial-gradient(circle, rgba(67,69,112,1) 3%, rgba(35,36,57,1) 60%);
            min-height: 100vh;
            margin: 0;
            padding: 1rem;
        }

        .detector-container {
            width: 100%;
            margin: 0;
            background: rgba(255,255,255,0.1);
            border-radius: 12px;
            padding: 1.5rem;
            border: 2px solid transparent;
            transition: all 0.3s ease;
            box-sizing: border-box;
        }

        .detector-container:hover {
            border-color: #29b6f6;
            box-shadow: 0 10px 25px rgba(41, 182, 246, 0.3);
        }

        .header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .title {
            color: #fff;
            font-size: 2.5rem;
            font-family: "Tilt Neon", sans-serif;
            margin-bottom: 0.5rem;
        }

        .subtitle {
            color: #ddd;
            font-size: 1.1rem;
            margin-bottom: 2rem;
        }

        .info-panel {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            border-left: 4px solid #4fc3f7;
        }

        .info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            padding: 0.5rem 0;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }

        .info-row:last-child {
            border-bottom: none;
            margin-bottom: 0;
        }

        .info-label {
            color: #4fc3f7;
            font-weight: 600;
            font-size: 1rem;
        }

        .info-value {
            color: #fff;
            font-size: 1rem;
            font-weight: 500;
        }

        .status-exists {
            color: #059669;
            background: rgba(5, 150, 105, 0.1);
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            border: 1px solid rgba(5, 150, 105, 0.3);
        }

        .status-not-exists {
            color: #dc2626;
            background: rgba(220, 38, 38, 0.1);
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            border: 1px solid rgba(220, 38, 38, 0.3);
        }

        .file-content-panel {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }

        .panel-title {
            color: #4fc3f7;
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .file-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
        }

        .stat-card {
            background: rgba(255,255,255,0.05);
            border-radius: 6px;
            padding: 1rem;
            text-align: center;
            border: 1px solid rgba(255,255,255,0.1);
        }

        .stat-value {
            color: #fff;
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
        }

        .stat-label {
            color: #bbb;
            font-size: 0.9rem;
        }

        .content-preview {
            background: #1a1a1a;
            border: 1px solid #333;
            border-radius: 6px;
            padding: 1rem;
            font-family: 'Courier New', monospace;
            font-size: 0.85rem;
            color: #e0e0e0;
            max-height: 300px;
            overflow-y: auto;
            white-space: pre-wrap;
            word-break: break-all;
        }

        .no-file-message {
            text-align: center;
            color: #888;
            padding: 2rem;
            font-style: italic;
        }

        .back-button {
            position: fixed;
            top: 2rem;
            left: 2rem;
            z-index: 1000;
        }

        .action-buttons {
            display: flex;
            gap: 1rem;
            justify-content: center;
            margin-top: 2rem;
        }

        .btn-action {
            padding: 1rem 2rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 1rem;
            text-decoration: none;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #fff;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        .btn-success {
            background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
            color: #fff;
        }

        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(76, 175, 80, 0.4);
        }

        .btn-secondary {
            background: rgba(255,255,255,0.1);
            color: #fff;
            border: 2px solid #29b6f6;
        }

        .btn-secondary:hover {
            background: #29b6f6;
            transform: translateY(-2px);
        }

        .sections-panel {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }

        .sections-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }

        .section-card {
            background: rgba(255,255,255,0.08);
            border: 1px solid rgba(255,255,255,0.1);
            border-radius: 8px;
            padding: 1rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            transition: all 0.3s ease;
        }

        .section-card:hover {
            background: rgba(255,255,255,0.12);
            border-color: #4fc3f7;
            transform: translateY(-2px);
        }

        .section-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #4fc3f7 0%, #29b6f6 100%);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.2rem;
        }

        /* Colores para las secciones del panel superior */
        .section-card[data-section="titulo-1"] .section-icon {
            background: linear-gradient(135deg, #9c27b0 0%, #7b1fa2 100%);
        }

        .section-card[data-section="aprendizaje-esperado"] .section-icon {
            background: linear-gradient(135deg, #4caf50 0%, #388e3c 100%);
        }

        .section-card[data-section="criterios-evaluacion"] .section-icon {
            background: linear-gradient(135deg, #ff5722 0%, #d84315 100%);
        }

        .section-card[data-section="contenidos"] .section-icon {
            background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%);
        }

        .section-card[data-section="archivos-pdf"] .section-icon {
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
        }

        .section-card[data-section="seccion-video"] .section-icon {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
        }

        .section-card[data-section="seccion-podcast"] .section-icon {
            background: linear-gradient(135deg, #607d8b 0%, #455a64 100%);
        }

        .section-card[data-section="seccion-cargar-archivo"] .section-icon {
            background: linear-gradient(135deg, #e91e63 0%, #c2185b 100%);
        }

        .section-card[data-section="respuesta-larga"] .section-icon {
            background: linear-gradient(135deg, #00bcd4 0%, #0097a7 100%);
        }

        .section-card[data-section="seccion-foro"] .section-icon {
            background: linear-gradient(135deg, #8bc34a 0%, #689f38 100%);
        }

        .section-card[data-section="termino-actividad"] .section-icon {
            background: linear-gradient(135deg, #795548 0%, #5d4037 100%);
        }

        .section-card[data-section="ruta-ejercitacion"] .section-icon {
            background: linear-gradient(135deg, #9e9e9e 0%, #757575 100%);
        }

        .section-card[data-section="geogebra-applet"] .section-icon {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
        }

        .section-info {
            flex: 1;
        }

        .section-info h4 {
            color: #fff;
            margin: 0 0 0.25rem 0;
            font-size: 1rem;
            font-weight: 600;
        }

        .section-info p {
            color: #bbb;
            margin: 0;
            font-size: 0.85rem;
        }

        .section-controls {
            display: flex;
            gap: 0.5rem;
        }

        .btn-add-section {
            background: linear-gradient(135deg, #ffffff 0%, #f5f5f5 100%);
            border: 2px solid #333333;
            border-radius: 6px;
            color: #000000;
            width: 35px;
            height: 35px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: bold;
        }

        .btn-add-section:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.2);
            background: linear-gradient(135deg, #f0f0f0 0%, #e0e0e0 100%);
            border-color: #000000;
        }

        .added-sections-panel {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }

        .added-sections-list {
            margin-top: 1rem;
        }

        .no-sections-message {
            text-align: center;
            color: #888;
            padding: 2rem;
            font-style: italic;
        }

        .no-sections-message i {
            font-size: 2rem;
            margin-bottom: 1rem;
            display: block;
        }

        .added-section-item {
            background: rgba(255,255,255,0.08);
            border: 1px solid rgba(255,255,255,0.1);
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            cursor: default;
            transition: all 0.3s ease;
            user-select: none;
        }

        .added-section-item:hover {
            background: rgba(255,255,255,0.12);
            border-color: #4fc3f7;
            transform: translateY(-1px);
        }

        .added-section-item.dragging {
            opacity: 0.6;
            transform: rotate(2deg) scale(1.02);
            cursor: grabbing;
            z-index: 1000;
            box-shadow: 0 10px 25px rgba(0,0,0,0.3);
        }

        .added-section-item.drag-over {
            border-color: #4fc3f7;
            background: rgba(79, 195, 247, 0.15);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(79, 195, 247, 0.2);
        }

        .drag-handle {
            color: #bbb;
            cursor: grab;
            padding: 0.5rem;
            border-radius: 4px;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .drag-handle:hover {
            color: #4fc3f7;
            background: rgba(79, 195, 247, 0.15);
            transform: scale(1.1);
        }

        .drag-handle:active {
            cursor: grabbing;
            color: #29b6f6;
        }

        .order-number {
            width: 35px;
            height: 35px;
            background: linear-gradient(135deg, #4fc3f7 0%, #29b6f6 100%);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 1rem;
            flex-shrink: 0;
            box-shadow: 0 2px 8px rgba(79, 195, 247, 0.3);
            transition: all 0.3s ease;
        }

        .added-section-item:hover .order-number {
            transform: scale(1.1);
            box-shadow: 0 4px 12px rgba(79, 195, 247, 0.4);
        }

        .added-section-item.dragging .order-number {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            box-shadow: 0 4px 15px rgba(255, 152, 0, 0.4);
        }

        .added-section-item .section-icon {
            width: 40px;
            height: 40px;
            font-size: 1rem;
        }

        /* Colores específicos para cada tipo de sección */
        .section-icon.titulo-1 {
            background: linear-gradient(135deg, #9c27b0 0%, #7b1fa2 100%);
        }

        .section-icon.aprendizaje-esperado {
            background: linear-gradient(135deg, #4caf50 0%, #388e3c 100%);
        }

        .section-icon.criterios-evaluacion {
            background: linear-gradient(135deg, #ff5722 0%, #d84315 100%);
        }

        .section-icon.contenidos {
            background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%);
        }

        .section-icon.archivos-pdf {
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
        }

        .section-icon.seccion-video {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
        }

        .section-icon.seccion-podcast {
            background: linear-gradient(135deg, #607d8b 0%, #455a64 100%);
        }

        .section-icon.seccion-cargar-archivo {
            background: linear-gradient(135deg, #e91e63 0%, #c2185b 100%);
        }

        .section-icon.respuesta-larga {
            background: linear-gradient(135deg, #00bcd4 0%, #0097a7 100%);
        }

        .section-icon.seccion-foro {
            background: linear-gradient(135deg, #8bc34a 0%, #689f38 100%);
        }

        .section-icon.termino-actividad {
            background: linear-gradient(135deg, #795548 0%, #5d4037 100%);
        }

        .section-icon.ruta-ejercitacion {
            background: linear-gradient(135deg, #9e9e9e 0%, #757575 100%);
        }

        .section-icon.geogebra-applet {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            font-weight: 700;
            font-size: 1.2rem;
        }

        .added-section-item .section-info h4 {
            font-size: 0.9rem;
        }

        .added-section-item .section-info p {
            font-size: 0.8rem;
        }

        .section-actions {
            display: flex;
            gap: 0.5rem;
        }

        .btn-edit-section {
            background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%);
            border: none;
            border-radius: 4px;
            color: white;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-right: 0.5rem;
        }

        .btn-edit-section:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(33, 150, 243, 0.4);
        }

        .btn-view-section {
            background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
            border: none;
            border-radius: 4px;
            color: white;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-right: 0.5rem;
        }

        .btn-view-section:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(76, 175, 80, 0.4);
        }

        .btn-remove-section {
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
            border: none;
            border-radius: 4px;
            color: white;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-remove-section:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(244, 67, 54, 0.4);
        }

        .btn-generate-vars {
            background: linear-gradient(135deg, #9c27b0 0%, #7b1fa2 100%);
            border: none;
            border-radius: 4px;
            color: white;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-generate-vars:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(156, 39, 176, 0.4);
        }

        .btn-move-section {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            border: none;
            border-radius: 4px;
            color: white;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-move-section:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(255, 152, 0, 0.4);
        }

        .btn-move-section:disabled {
            background: #666;
            cursor: not-allowed;
            opacity: 0.5;
        }

        .btn-move-section:disabled:hover {
            transform: none;
            box-shadow: none;
        }

        .move-controls {
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
        }


        @media (max-width: 768px) {
            body {
                padding: 1rem;
            }
            
            .sections-grid {
                grid-template-columns: 1fr;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .back-button {
                position: static;
                margin-bottom: 1rem;
            }
        }

        /* Estilos para el modal de edición */
        .modal-edicion {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            backdrop-filter: blur(5px);
        }

        .modal-content {
            background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            border-radius: 12px;
            width: 90%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            border: 2px solid #4fc3f7;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px 12px 0 0;
        }

        .modal-header h3 {
            color: #4fc3f7;
            margin: 0;
            font-size: 1.3rem;
            font-weight: 600;
        }

        .btn-cerrar-modal {
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
            border: none;
            border-radius: 50%;
            color: white;
            width: 35px;
            height: 35px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-cerrar-modal:hover {
            transform: scale(1.1);
            box-shadow: 0 3px 10px rgba(244, 67, 54, 0.4);
        }

        .modal-body {
            padding: 2rem;
        }

        .modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: 1rem;
            padding: 1.5rem;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
            background: rgba(255, 255, 255, 0.05);
            border-radius: 0 0 12px 12px;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            color: #4fc3f7;
            font-weight: 600;
            margin-bottom: 0.5rem;
            font-size: 0.95rem;
        }

        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 0.75rem;
            border: 2px solid rgba(255, 255, 255, 0.1);
            border-radius: 6px;
            background: rgba(255, 255, 255, 0.1);
            color: #fff;
            font-size: 1rem;
            transition: all 0.3s ease;
            box-sizing: border-box;
        }

        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: #4fc3f7;
            background: rgba(255, 255, 255, 0.15);
            box-shadow: 0 0 10px rgba(79, 195, 247, 0.3);
        }

        .form-group input::placeholder,
        .form-group textarea::placeholder {
            color: rgba(255, 255, 255, 0.6);
        }

        .form-group textarea {
            min-height: 100px;
            resize: vertical;
        }

        .readonly-field {
            background: rgba(255, 255, 255, 0.05);
            border: 2px solid rgba(255, 255, 255, 0.1);
            border-radius: 6px;
            padding: 0.75rem;
            color: #fff;
            font-size: 1rem;
            min-height: 40px;
            display: flex;
            align-items: center;
            border-left: 4px solid #4caf50;
        }

        .readonly-field i {
            color: #4caf50;
            margin-right: 0.5rem;
        }

        .info-box {
            background-color: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 4px;
            padding: 12px;
            margin: 8px 0;
            color: #1976d2;
        }

        .criterios-readonly {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
        }

        .criterio-readonly-item {
            display: flex;
            gap: 1rem;
            align-items: flex-start;
            background: rgba(255, 255, 255, 0.03);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 6px;
            padding: 0.75rem;
            border-left: 4px solid #4caf50;
        }

        .criterio-codigo-readonly {
            background: rgba(76, 175, 80, 0.2);
            color: #4caf50;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-weight: 600;
            font-size: 0.9rem;
            min-width: 80px;
            text-align: center;
            flex-shrink: 0;
        }

        .criterio-descripcion-readonly {
            color: #fff;
            flex: 1;
            line-height: 1.4;
        }

        .contenidos-readonly {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }

        .contenido-readonly-item {
            display: flex;
            gap: 1rem;
            align-items: flex-start;
            background: rgba(255, 255, 255, 0.03);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            padding: 1rem;
            border-left: 4px solid #4caf50;
        }

        .contenido-icono-readonly {
            background: rgba(76, 175, 80, 0.2);
            color: #4caf50;
            width: 50px;
            height: 50px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            flex-shrink: 0;
        }

        .contenido-info-readonly {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .contenido-titulo-readonly {
            color: #4caf50;
            font-weight: 600;
            font-size: 1.1rem;
            margin-bottom: 0.25rem;
        }

        .contenido-descripcion-readonly {
            color: #fff;
            line-height: 1.4;
            opacity: 0.9;
        }

        /* Estilos para el panel de vista previa */
        .preview-panel {
            background: rgba(255,255,255,0.05);
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            border: 2px solid rgba(79, 195, 247, 0.3);
            box-shadow: 0 8px 25px rgba(79, 195, 247, 0.15);
        }

        .preview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .preview-actions {
            display: flex;
            gap: 0.75rem;
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
        }

        .preview-container {
            position: relative;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            min-height: 600px;
        }

        .preview-loading {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.95);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10;
        }

        .loading-spinner {
            text-align: center;
            color: #4fc3f7;
        }

        .loading-spinner i {
            font-size: 2rem;
            margin-bottom: 1rem;
            display: block;
        }

        .loading-spinner p {
            margin: 0;
            font-size: 1rem;
            font-weight: 500;
        }

        .preview-container iframe {
            transition: opacity 0.3s ease;
        }

        .preview-container.loading iframe {
            opacity: 0.3;
        }
    </style>
</head>

<body>
    <!-- Botón de regreso -->
    <div class="back-button">
        <button onclick="volverAlPanel()" class="boton-azul">
            <i class="fas fa-arrow-left"></i>
            <span>Volver al Panel</span>
        </button>
    </div>

    <div class="detector-container">
        <!-- Header -->
        <div class="header">
            <h1 class="title">
                <i class="fas fa-edit"></i>
                Crear/Editar Actividad
            </h1>
        </div>

        <!-- Información de la Actividad -->
        <div class="info-panel">
            <div class="info-row">
                <span class="info-label">
                    <i class="fas fa-file-code"></i> Nombre del Archivo:
                </span>
                <span class="info-value"><?php echo $nombre_archivo ? $nombre_archivo : 'No generado'; ?></span>
                <span class="info-label" style="margin-left: 2rem;">
                    <i class="fas fa-info-circle"></i> Estado:
                </span>
                <span class="info-value">
                    <?php if ($nombre_archivo): ?>
                        <?php if ($archivo_existe): ?>
                            <span class="status-exists">
                                <i class="fas fa-check-circle"></i> Archivo Existe
                                <?php if (count($secciones_detectadas) > 0): ?>
                                    <br><small style="color: #059669; font-size: 0.8rem;">
                                        <i class="fas fa-magic"></i> <?php echo count($secciones_detectadas); ?> sección(es) detectada(s)
                                    </small>
                                <?php endif; ?>
                            </span>
                        <?php else: ?>
                            <span class="status-not-exists">
                                <i class="fas fa-times-circle"></i> Archivo No Encontrado
                            </span>
                        <?php endif; ?>
                    <?php else: ?>
                        <span class="status-not-exists">
                            <i class="fas fa-exclamation-triangle"></i> Parámetros Incompletos
                        </span>
                    <?php endif; ?>
                </span>
            </div>
        </div>

        <!-- Panel de Selección de Secciones -->
        <div class="sections-panel">
            <h3 class="panel-title">
                <i class="fas fa-puzzle-piece"></i>
                Seleccionar Secciones para la Actividad
            </h3>
            
            <div class="sections-grid">
                <div class="section-card" data-section="titulo-1">
                    <div class="section-icon">
                        <i class="fas fa-heading"></i>
                    </div>
                    <div class="section-info">
                        <h4>Título de la Actividad</h4>
                        <p>Encabezado principal de la actividad</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('titulo-1')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="aprendizaje-esperado">
                    <div class="section-icon">
                        <i class="fas fa-bullseye"></i>
                    </div>
                    <div class="section-info">
                        <h4>Aprendizaje Esperado</h4>
                        <p>Objetivos de aprendizaje de la actividad</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('aprendizaje-esperado')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="criterios-evaluacion">
                    <div class="section-icon">
                        <i class="fas fa-clipboard-check"></i>
                    </div>
                    <div class="section-info">
                        <h4>Criterios de Evaluación</h4>
                        <p>Criterios para evaluar el desempeño</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('criterios-evaluacion')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="contenidos">
                    <div class="section-icon">
                        <i class="fas fa-list-ul"></i>
                    </div>
                    <div class="section-info">
                        <h4>Contenidos</h4>
                        <p>Lista de contenidos con iconos</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('contenidos')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="archivos-pdf">
                    <div class="section-icon">
                        <i class="fas fa-file-pdf"></i>
                    </div>
                    <div class="section-info">
                        <h4>Documento PDF</h4>
                        <p>Vista previa y descarga de PDF</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('archivos-pdf')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="seccion-video">
                    <div class="section-icon">
                        <i class="fas fa-play-circle"></i>
                    </div>
                    <div class="section-info">
                        <h4>Video Educativo</h4>
                        <p>Reproductor de video YouTube/Vimeo</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('seccion-video')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="seccion-podcast">
                    <div class="section-icon">
                        <i class="fas fa-microphone"></i>
                    </div>
                    <div class="section-info">
                        <h4>Podcast Educativo</h4>
                        <p>Reproductor de audio personalizado</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('seccion-podcast')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="seccion-cargar-archivo">
                    <div class="section-icon">
                        <i class="fas fa-upload"></i>
                    </div>
                    <div class="section-info">
                        <h4>Cargar Archivo</h4>
                        <p>Subida de documentos con drag & drop</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('seccion-cargar-archivo')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="respuesta-larga">
                    <div class="section-icon">
                        <i class="fas fa-edit"></i>
                    </div>
                    <div class="section-info">
                        <h4>Respuesta Larga</h4>
                        <p>Área de texto para respuesta del estudiante</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('respuesta-larga')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="seccion-foro">
                    <div class="section-icon">
                        <i class="fas fa-comments"></i>
                    </div>
                    <div class="section-info">
                        <h4>Foro de Opinión</h4>
                        <p>Discusión y comentarios de estudiantes</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('seccion-foro')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="termino-actividad">
                    <div class="section-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="section-info">
                        <h4>Término de Actividad</h4>
                        <p>Botón para marcar actividad como completada</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('termino-actividad')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="texto-rico">
                    <div class="section-icon">
                        <i class="fas fa-square-root-alt"></i>
                    </div>
                    <div class="section-info">
                        <h4>Texto con Fórmulas</h4>
                        <p>Editor de texto con fórmulas matemáticas e imágenes</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('texto-rico')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="ruta-ejercitacion">
                    <div class="section-icon">
                        <i class="fas fa-route"></i>
                    </div>
                    <div class="section-info">
                        <h4>Ruta de Ejercitación Progresiva</h4>
                        <p>Ejercicios interactivos progresivos desde archivo HTML</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('ruta-ejercitacion')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>

                <div class="section-card" data-section="geogebra-applet">
                    <div class="section-icon" style="background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%); font-weight: 700; font-size: 1.2rem;">
                        GG
                    </div>
                    <div class="section-info">
                        <h4>Applet GeoGebra</h4>
                        <p>Applet interactivo de GeoGebra incrustado en la actividad</p>
                    </div>
                    <div class="section-controls">
                        <button class="btn-add-section" onclick="agregarSeccion('geogebra-applet')">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Panel de Secciones Agregadas -->
        <div class="added-sections-panel">
            <h3 class="panel-title">
                <i class="fas fa-list"></i>
                Secciones Agregadas (<span id="section-count">0</span>)
            </h3>
            <div id="added-sections-list" class="added-sections-list">
                <div class="no-sections-message">
                    <i class="fas fa-info-circle"></i>
                    <p>No hay secciones agregadas. Selecciona secciones del panel superior.</p>
                </div>
            </div>
        </div>

        <!-- Vista Previa en Línea -->
        <?php if ($archivo_existe): ?>
            <div class="preview-panel">
                <div class="preview-header">
                    <h3 class="panel-title">
                        <i class="fas fa-eye"></i>
                        Vista Previa de la Actividad
                    </h3>
                    <div class="preview-actions">
                        <button class="btn-action btn-secondary btn-sm" onclick="abrirVistaPreviaExterna()">
                    <i class="fas fa-external-link-alt"></i>
                            Abrir en Nueva Ventana
                </button>
                        <button class="btn-action btn-secondary btn-sm" onclick="actualizarVistaPrevia()">
                            <i class="fas fa-sync-alt"></i>
                            Actualizar
                        </button>
                        <?php if ($archivo_existe): ?>
                            <button class="btn-action btn-primary btn-sm" onclick="generarArchivo()" id="btn-generar">
                                <i class="fas fa-save"></i>
                                Guardar Cambios
                            </button>
                        <?php else: ?>
                            <button class="btn-action btn-success btn-sm" onclick="generarArchivo()" id="btn-generar">
                                <i class="fas fa-save"></i>
                                Crear Actividad
                </button>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="preview-container">
                    <iframe 
                        id="preview-iframe" 
                        src="../curso.php?curso=<?php echo urlencode($curso); ?>&unidad=<?php echo urlencode($unidad); ?>&recurso=<?php echo urlencode($actividad); ?>&preview=1"
                        width="100%" 
                        height="600"
                        style="border: none; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);"
                        title="Vista previa de la actividad">
                    </iframe>
                    <div class="preview-loading" id="preview-loading" style="display: none;">
                        <div class="loading-spinner">
                            <i class="fas fa-spinner fa-spin"></i>
                            <p>Cargando vista previa...</p>
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
        <!-- Botones de Acción -->
        <div class="action-buttons">
            <button class="btn-action btn-secondary" onclick="recargarDetector()">
                <i class="fas fa-sync-alt"></i>
                Recargar Página
            </button>
        </div>

        <!-- Modal de Edición de Sección -->
        <div id="modal-edicion" class="modal-edicion" style="display: none;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3 id="modal-titulo">Editar Sección</h3>
                    <button class="btn-cerrar-modal" onclick="cerrarModalEdicion()">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <div class="modal-body" id="modal-body">
                    <!-- El contenido del editor se cargará aquí dinámicamente -->
                </div>
                <div class="modal-footer" id="modal-footer">
                    <button class="btn-action btn-secondary" onclick="cerrarModalEdicion()">
                        <i class="fas fa-times"></i>
                        Cerrar
                    </button>
                </div>
            </div>
        </div>

    </div>

    <script>
        // Información de la página actual
        const parametrosActuales = {
            curso: '<?php echo $curso; ?>',
            unidad: '<?php echo $unidad; ?>',
            actividad: '<?php echo $actividad; ?>',
            modo: '<?php echo $modo; ?>',
            nombreArchivo: '<?php echo $nombre_archivo; ?>',
            archivoExiste: <?php echo $archivo_existe ? 'true' : 'false'; ?>
        };

        // Secciones detectadas automáticamente del archivo existente
        const seccionesDetectadas = <?php echo json_encode($secciones_detectadas); ?>;
        
        // Número de actividad desde la URL (sin ceros iniciales)
        const numeroActividad = parseInt(<?php echo json_encode($actividad); ?>) || 1;

        // Configuración de secciones disponibles
        const seccionesDisponibles = {
            'titulo-1': {
                nombre: 'Título de la Actividad',
                descripcion: 'Encabezado principal de la actividad',
                icono: 'fas fa-heading'
            },
            'aprendizaje-esperado': {
                nombre: 'Aprendizaje Esperado',
                descripcion: 'Objetivos de aprendizaje de la actividad',
                icono: 'fas fa-bullseye'
            },
            'criterios-evaluacion': {
                nombre: 'Criterios de Evaluación',
                descripcion: 'Criterios para evaluar el desempeño',
                icono: 'fas fa-clipboard-check'
            },
            'contenidos': {
                nombre: 'Contenidos',
                descripcion: 'Lista de contenidos con iconos',
                icono: 'fas fa-list-ul'
            },
            'archivos-pdf': {
                nombre: 'Documento PDF',
                descripcion: 'Guía y material de apoyo en formato PDF',
                icono: 'fas fa-file-pdf'
            },
            'seccion-video': {
                nombre: 'Video Educativo',
                descripcion: 'Reproductor de video YouTube/Vimeo',
                icono: 'fas fa-play-circle'
            },
            'seccion-podcast': {
                nombre: 'Podcast Educativo',
                descripcion: 'Reproductor de audio personalizado',
                icono: 'fas fa-microphone'
            },
            'seccion-cargar-archivo': {
                nombre: 'Cargar Archivo',
                descripcion: 'Subida de documentos con drag & drop',
                icono: 'fas fa-upload'
            },
            'respuesta-larga': {
                nombre: 'Respuesta Larga',
                descripcion: 'Área de texto para respuesta del estudiante',
                icono: 'fas fa-edit'
            },
            'seccion-foro': {
                nombre: 'Foro de Opinión',
                descripcion: 'Discusión y comentarios de estudiantes',
                icono: 'fas fa-comments'
            },
            'termino-actividad': {
                nombre: 'Término de Actividad',
                descripcion: 'Botón para marcar actividad como completada',
                icono: 'fas fa-check-circle'
            },
            'texto-rico': {
                nombre: 'Texto con Fórmulas',
                descripcion: 'Editor de texto con fórmulas matemáticas e imágenes',
                icono: 'fas fa-square-root-alt'
            },
            'ruta-ejercitacion': {
                nombre: 'Ruta de Ejercitación Progresiva',
                descripcion: 'Ejercicios interactivos progresivos desde archivo HTML',
                icono: 'fas fa-route'
            },
            'geogebra-applet': {
                nombre: 'Applet GeoGebra',
                descripcion: 'Applet interactivo de GeoGebra incrustado en la actividad',
                icono: 'fas fa-square-root-alt'
            }
        };

        // Array para almacenar las secciones agregadas
        let seccionesAgregadas = [];
        
        // Variable para almacenar la sección que se está editando
        let seccionEditando = null;

        console.log('Parámetros actuales:', parametrosActuales);

        // Función para obtener el siguiente número de respuesta larga
        function obtenerSiguienteNumeroRespuestaLarga() {
            const respuestasLargas = seccionesAgregadas.filter(s => s.tipo === 'respuesta-larga');
            if (respuestasLargas.length === 0) {
                return 1;
            }
            
            // Obtener el número más alto y sumar 1
            const numeros = respuestasLargas.map(s => s.numero_respuesta || 0);
            return Math.max(...numeros) + 1;
        }

        // Función para generar variables de base de datos para Respuesta Larga
        async function generarVariablesRespuestaLarga(seccionId) {
            const seccion = seccionesAgregadas.find(s => s.id === seccionId);
            if (!seccion) {
                alert('Error: Sección no encontrada');
                return;
            }

            // Obtener parámetros de la URL
            const curso = parametrosActuales.curso;
            const unidad = parametrosActuales.unidad;
            const actividad = parametrosActuales.actividad;
            
            if (!curso || !unidad || !actividad) {
                alert('Error: No se pudieron obtener los parámetros del curso, unidad y actividad');
                return;
            }

            // Obtener el número de respuesta larga
            const numeroRespuesta = seccion.numero_respuesta || 1;
            
            // Mostrar confirmación con detalles
            const confirmacion = confirm(
                `¿Generar variables de base de datos para esta Respuesta Larga?\n\n` +
                `📊 Detalles:\n` +
                `• Curso: ${curso}\n` +
                `• Unidad: ${unidad}\n` +
                `• Actividad: ${actividad}\n` +
                `• Respuesta Larga: ${numeroRespuesta}\n\n` +
                `🗄️ Variables a crear:\n` +
                `• RL${numeroRespuesta.toString().padStart(2, '0')}ACT${actividad.padStart(2, '0')} (Respuesta del estudiante)\n` +
                `• RTRL${numeroRespuesta.toString().padStart(2, '0')}ACT${actividad.padStart(2, '0')} (Retroalimentación)\n` +
                `• NTRL${numeroRespuesta.toString().padStart(2, '0')}ACT${actividad.padStart(2, '0')} (Nota)\n\n` +
                `📋 Tabla: avance_${curso}_u${unidad.replace('Unidad ', '').padStart(2, '0')}`
            );

            if (!confirmacion) {
                return;
            }

            try {
                const response = await fetch('generar_variables_respuesta_larga.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        curso: curso,
                        unidad: unidad,
                        actividad: actividad,
                        numero_respuesta: numeroRespuesta
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    alert(
                        '✅ Variables generadas exitosamente!\n\n' +
                        `📋 Tabla: ${result.details.tabla}\n` +
                        `🔧 Columnas creadas:\n` +
                        result.details.columnas_creadas.join('\n') + '\n\n' +
                        `ℹ️ ${result.details.mensaje}`
                    );
                } else {
                    alert(`❌ Error al generar variables:\n${result.message}`);
                }
            } catch (error) {
                console.error('Error:', error);
                alert('❌ Error de conexión al generar variables');
            }
        }

        // Función para reasignar números de respuesta larga
        function reasignarNumerosRespuestaLarga() {
            const respuestasLargas = seccionesAgregadas.filter(s => s.tipo === 'respuesta-larga');
            respuestasLargas.forEach((seccion, index) => {
                seccion.numero_respuesta = index + 1;
                seccion.nombre = `Respuesta Larga ${(index + 1).toString().padStart(2, '0')}`;
            });
        }

        function agregarSeccion(tipoSeccion) {
            if (seccionesDisponibles[tipoSeccion]) {
                const seccion = {
                    id: Date.now() + Math.random(), // ID único
                    tipo: tipoSeccion,
                    nombre: seccionesDisponibles[tipoSeccion].nombre,
                    descripcion: seccionesDisponibles[tipoSeccion].descripcion,
                    icono: seccionesDisponibles[tipoSeccion].icono
                };
                
                // Si es respuesta larga, asignar número automático
                if (tipoSeccion === 'respuesta-larga') {
                    const numero = obtenerSiguienteNumeroRespuestaLarga();
                    seccion.numero_respuesta = numero;
                    seccion.nombre = `Respuesta Larga ${numero.toString().padStart(2, '0')}`;
                }
                
                // Si es término de actividad, asignar contenido inicial
                if (tipoSeccion === 'termino-actividad') {
                    const actividadNumero = <?php echo isset($_GET['actividad']) ? (int)$_GET['actividad'] : 1; ?>;
                    seccion.contenido = {
                        texto_boton: `Terminar Actividad ${actividadNumero.toString().padStart(2, '0')}`,
                        variable_condicion: '',
                        mensaje_exito: '¡Actividad completada exitosamente!'
                    };
                }
                
                // Si es texto rico, asignar contenido inicial vacío
                if (tipoSeccion === 'texto-rico') {
                    seccion.contenido = {
                        titulo: '',
                        contenido: ''
                    };
                }
                
                // Si es ruta de ejercitación, asignar contenido inicial vacío
                if (tipoSeccion === 'ruta-ejercitacion') {
                    seccion.contenido = {
                        titulo: '',
                        archivo: null
                    };
                }
                
                // Si es GeoGebra applet, asignar contenido inicial vacío
                if (tipoSeccion === 'geogebra-applet') {
                    seccion.contenido = {
                        titulo: '',
                        url: ''
                    };
                }
                
                // Si es aprendizaje esperado, cargar contenido desde configuración
                if (tipoSeccion === 'aprendizaje-esperado') {
                    seccion.contenido = <?php echo json_encode(obtenerAprendizajeEsperadoConfiguracion($curso, $unidad)); ?>;
                    seccion.solo_lectura = true; // Marcar como solo lectura
                }
                
                // Si es criterios de evaluación, cargar contenido desde configuración
                if (tipoSeccion === 'criterios-evaluacion') {
                    seccion.contenido = <?php echo json_encode(obtenerCriteriosEvaluacionConfiguracion($curso, $unidad)); ?>;
                    seccion.solo_lectura = true; // Marcar como solo lectura
                }
                
                // Si es contenidos, cargar contenido desde configuración
                if (tipoSeccion === 'contenidos') {
                    seccion.contenido = <?php echo json_encode(obtenerContenidosConfiguracion($curso, $unidad)); ?>;
                    seccion.solo_lectura = true; // Marcar como solo lectura
                }
                
                seccionesAgregadas.push(seccion);
                actualizarListaSecciones();
                console.log('Sección agregada:', seccion);
            }
        }

        function quitarSeccion(idSeccion) {
            // Verificar si la sección que se va a eliminar es una respuesta larga
            const seccionEliminada = seccionesAgregadas.find(seccion => seccion.id == idSeccion);
            const esRespuestaLarga = seccionEliminada && seccionEliminada.tipo === 'respuesta-larga';
            
            // Eliminar la sección
            seccionesAgregadas = seccionesAgregadas.filter(seccion => seccion.id !== idSeccion);
            
            // Si se eliminó una respuesta larga, reasignar números
            if (esRespuestaLarga) {
                reasignarNumerosRespuestaLarga();
            }
            
            actualizarListaSecciones();
            console.log('Sección removida:', idSeccion);
        }

        function actualizarListaSecciones() {
            const lista = document.getElementById('added-sections-list');
            const contador = document.getElementById('section-count');
            
            contador.textContent = seccionesAgregadas.length;
            
            if (seccionesAgregadas.length === 0) {
                lista.innerHTML = `
                    <div class="no-sections-message">
                        <i class="fas fa-info-circle"></i>
                        <p>No hay secciones agregadas. Selecciona secciones del panel superior.</p>
                    </div>
                `;
            } else {
                lista.innerHTML = seccionesAgregadas.map((seccion, index) => `
                    <div class="added-section-item" data-id="${seccion.id}">
                        <div class="order-number">${index + 1}</div>
                        <div class="drag-handle" draggable="true" title="Arrastrar para reordenar">
                            <i class="fas fa-grip-vertical"></i>
                        </div>
                        <div class="section-icon ${seccion.tipo}">
                            ${seccion.icono && seccion.icono.includes('fa-') ? `<i class="${seccion.icono}"></i>` : seccion.icono || '<i class="fas fa-question"></i>'}
                        </div>
                        <div class="section-info">
                            <h4>${seccion.nombre}</h4>
                            <p>${seccion.descripcion}</p>
                        </div>
                        <div class="section-actions">
                            ${seccion.solo_lectura ? 
                                `<button class="btn-view-section" onclick="verSeccion(${seccion.id})" title="Ver contenido (solo lectura)">
                                    <i class="fas fa-eye"></i>
                                </button>` :
                                `<button class="btn-edit-section" onclick="editarSeccion(${seccion.id})" title="Editar sección">
                                    <i class="fas fa-edit"></i>
                                </button>`
                            }
                            ${seccion.tipo === 'respuesta-larga' ? 
                                `<button class="btn-generate-vars" onclick="generarVariablesRespuestaLarga(${seccion.id})" title="Generar variables de base de datos">
                                    <i class="fas fa-database"></i>
                                </button>` : ''
                            }
                            ${seccion.tipo === 'termino-actividad' ? 
                                `<button class="btn-generate-vars" onclick="abrirVerBD()" title="Ver variables de base de datos">
                                    <i class="fas fa-table"></i>
                                </button>` : ''
                            }
                            <button class="btn-remove-section" onclick="quitarSeccion(${seccion.id})" title="Quitar sección">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                `).join('');
                
                // Inicializar drag & drop después de actualizar el HTML
                inicializarDragAndDrop();
            }
        }

        function inicializarDragAndDrop() {
            const handles = document.querySelectorAll('.drag-handle');
            const items = document.querySelectorAll('.added-section-item');
            
            console.log('Inicializando drag and drop para', handles.length, 'handles');
            
            // Agregar listeners a los handles (para dragstart)
            handles.forEach(handle => {
                handle.addEventListener('dragstart', handleDragStart);
                handle.addEventListener('dragend', handleDragEnd);
            });
            
            // Agregar listeners a los items (para dragover y drop)
            items.forEach(item => {
                item.addEventListener('dragover', handleDragOver);
                item.addEventListener('drop', handleDrop);
            });
        }

        let draggedElement = null;

        function handleDragStart(e) {
            // El handle está siendo arrastrado, pero queremos trabajar con el item padre
            draggedElement = this.closest('.added-section-item');
            draggedElement.classList.add('dragging');
            
            e.dataTransfer.effectAllowed = 'move';
            e.dataTransfer.setData('text/plain', draggedElement.dataset.id);
            
            console.log('Iniciando drag de elemento:', draggedElement.dataset.id);
        }

        function handleDragOver(e) {
                e.preventDefault();
            e.dataTransfer.dropEffect = 'move';
            
            // Remover clase drag-over de todos los elementos
            document.querySelectorAll('.added-section-item').forEach(item => {
                item.classList.remove('drag-over');
            });
            
            // Agregar clase drag-over al elemento sobre el que se está arrastrando
            if (this !== draggedElement && !this.classList.contains('dragging')) {
                this.classList.add('drag-over');
            }
        }

        function handleDrop(e) {
            e.preventDefault();
                e.stopPropagation();
            
            console.log('handleDrop ejecutado');
            console.log('draggedElement:', draggedElement);
            console.log('target element:', this);
            
            if (draggedElement && draggedElement !== this) {
                const draggedId = parseFloat(draggedElement.dataset.id);
                const targetId = parseFloat(this.dataset.id);
                
                console.log('Drop: ID arrastrado =', draggedId, 'ID objetivo =', targetId);
                console.log('Secciones actuales:', seccionesAgregadas.map(s => ({id: s.id, tipo: s.tipo})));
                
                // Encontrar índices en el array - usar == en lugar de === para comparación más flexible
                const draggedIndex = seccionesAgregadas.findIndex(s => s.id == draggedId);
                const targetIndex = seccionesAgregadas.findIndex(s => s.id == targetId);
                
                console.log('Índices encontrados - arrastrado:', draggedIndex, 'objetivo:', targetIndex);
                
                if (draggedIndex !== -1 && targetIndex !== -1 && draggedIndex !== targetIndex) {
                    console.log('Realizando reordenamiento...');
                    
                    // Remover el elemento arrastrado
                    const [draggedSection] = seccionesAgregadas.splice(draggedIndex, 1);
                    
                    // Insertar en la nueva posición
                    seccionesAgregadas.splice(targetIndex, 0, draggedSection);
                    
                    // Actualizar la vista
                    actualizarListaSecciones();
                    
                    console.log('✅ Secciones reordenadas exitosamente:', seccionesAgregadas.map(s => ({id: s.id, tipo: s.tipo})));
                } else {
                    console.log('❌ No se pudo reordenar - índices inválidos o iguales');
                }
            } else {
                console.log('❌ No se pudo reordenar - elementos inválidos');
            }
            
            return false;
        }

        function handleDragEnd(e) {
            console.log('Drag finalizado');
            
            // Remover todas las clases de drag
            document.querySelectorAll('.added-section-item').forEach(item => {
                item.classList.remove('dragging', 'drag-over');
            });
            
            draggedElement = null;
        }

        function generarArchivo() {
            if (seccionesAgregadas.length === 0) {
                alert('Debes agregar al menos una sección antes de generar el archivo.');
                return;
            }

            console.log('Generando archivo con secciones:', seccionesAgregadas);
            
            // Mostrar mensaje de procesamiento
            const btnGenerar = document.getElementById('btn-generar');
            if (btnGenerar) {
                btnGenerar.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Generando...';
                btnGenerar.disabled = true;
            }

            // Enviar datos al servidor usando fetch
            const datos = {
                curso: parametrosActuales.curso,
                unidad: parametrosActuales.unidad,
                actividad: parametrosActuales.actividad,
                secciones: JSON.stringify(seccionesAgregadas),
                solo_servidor: true // Indicar que solo guarde en servidor
            };

            fetch('guardar_archivo_actividad.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams(datos)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    mostrarMensajeExito('¡Archivo guardado exitosamente en el servidor!');
                    
                    // Actualizar la vista previa si existe
                    if (document.getElementById('preview-iframe')) {
                        setTimeout(() => {
                            actualizarVistaPrevia();
                        }, 1000);
                    }
                    
                    // No recargar automáticamente, esperar a que el usuario presione "Volver al Panel"
                    // El estado se guardará cuando presione el botón
                } else {
                    mostrarMensajeError('Error al guardar el archivo: ' + (data.error || 'Error desconocido'));
                }
            })
            .catch(error => {
                console.error('Error:', error);
                mostrarMensajeError('Error de conexión al guardar el archivo');
            });

            // Restaurar botón después de un breve delay
            setTimeout(() => {
                if (btnGenerar) {
                    const textoOriginal = parametrosActuales.archivoExiste ? 
                        '<i class="fas fa-save"></i> Guardar Cambios' : 
                        '<i class="fas fa-save"></i> Crear Actividad';
                    btnGenerar.innerHTML = textoOriginal;
                    btnGenerar.disabled = false;
                }
            }, 2000);
        }

        function recargarDetector() {
            window.location.reload();
        }

        // Función para volver al panel manteniendo el estado
        function volverAlPanel() {
            // Guardar el estado actual de los despliegues en localStorage
            const expandedElements = document.querySelectorAll('.expanded');
            const expandedIds = Array.from(expandedElements).map(el => el.id);
            localStorage.setItem('panel_expanded_elements', JSON.stringify(expandedIds));
            
            // Ir al panel
            window.location.href = '../panel.php';
        }

        // Función para mostrar mensaje de éxito
        function mostrarMensajeExito(mensaje) {
            const mensajeDiv = document.createElement('div');
            mensajeDiv.className = 'mensaje-estado exito';
            mensajeDiv.innerHTML = `
                <div style="background: linear-gradient(135deg, #4caf50 0%, #45a049 100%); 
                           color: white; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;
                           display: flex; align-items: center; gap: 0.5rem; box-shadow: 0 4px 12px rgba(76, 175, 80, 0.3);">
                    <i class="fas fa-check-circle"></i>
                    <span>${mensaje}</span>
                </div>
            `;
            
            const contenedor = document.querySelector('.detector-container');
            contenedor.insertBefore(mensajeDiv, contenedor.firstChild);
            
            // Remover el mensaje después de 5 segundos
            setTimeout(() => {
                if (mensajeDiv.parentNode) {
                    mensajeDiv.parentNode.removeChild(mensajeDiv);
                }
            }, 5000);
        }

        // Función para mostrar mensaje de error
        function mostrarMensajeError(mensaje) {
            const mensajeDiv = document.createElement('div');
            mensajeDiv.className = 'mensaje-estado error';
            mensajeDiv.innerHTML = `
                <div style="background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%); 
                           color: white; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;
                           display: flex; align-items: center; gap: 0.5rem; box-shadow: 0 4px 12px rgba(244, 67, 54, 0.3);">
                    <i class="fas fa-exclamation-circle"></i>
                    <span>${mensaje}</span>
                </div>
            `;
            
            const contenedor = document.querySelector('.detector-container');
            contenedor.insertBefore(mensajeDiv, contenedor.firstChild);
            
            // Remover el mensaje después de 7 segundos
            setTimeout(() => {
                if (mensajeDiv.parentNode) {
                    mensajeDiv.parentNode.removeChild(mensajeDiv);
                }
            }, 7000);
        }

        // Función para abrir vista previa en ventana externa
        function abrirVistaPreviaExterna() {
            const urlCurso = `../curso.php?curso=${parametrosActuales.curso}&unidad=${encodeURIComponent(parametrosActuales.unidad)}&recurso=${parametrosActuales.actividad}&preview=1`;
            const opciones = 'width=1200,height=800,scrollbars=yes,resizable=yes,location=yes,toolbar=yes,menubar=yes';
            window.open(urlCurso, 'vista_previa_curso', opciones);
        }

        // Función para abrir editor de rutas
        function abrirEditorRutas() {
            // Extraer solo el número de la unidad (ej: "Unidad 01" -> "01")
            const unidadNumero = parametrosActuales.unidad.replace('Unidad ', '').padStart(2, '0');
            
            // Construir el nombre de la ruta basado en curso, unidad y actividad
            const nombreRuta = `ruta_${parametrosActuales.curso}_u${unidadNumero}_act${parametrosActuales.actividad.toString().padStart(2, '0')}`;
            const urlEditor = `../rutas/editor_rutas.php?curso=${parametrosActuales.curso}&unidad=${unidadNumero}&actividad=${parametrosActuales.actividad}&nombre_ruta=${nombreRuta}`;
            const opciones = 'width=1200,height=900,scrollbars=yes,resizable=yes,location=yes';
            window.open(urlEditor, 'editor_rutas', opciones);
        }

        // Función para abrir visor de base de datos
        function abrirVerBD() {
            // Extraer número de unidad
            const unidadNumero = parametrosActuales.unidad.replace('Unidad ', '').padStart(2, '0');
            const urlBD = `ver_base_datos.php?curso=${parametrosActuales.curso}&unidad=${unidadNumero}`;
            window.open(urlBD, '_blank', 'width=1200,height=800,scrollbars=yes,resizable=yes');
        }

        // Función para cargar variables de la base de datos en el select
        async function cargarVariablesBD() {
            const select = document.getElementById('termino-variable-condicion');
            if (!select) return;

            // Extraer número de unidad
            const unidadNumero = parametrosActuales.unidad.replace('Unidad ', '').padStart(2, '0');
            
            try {
                const response = await fetch(`obtener_variables_termino_actividad.php?curso=${parametrosActuales.curso}&unidad=${unidadNumero}`);
                const data = await response.json();
                
                if (data.success && data.variables) {
                    // Limpiar opciones existentes excepto la primera
                    select.innerHTML = '<option value="">Seleccione una variable...</option>';
                    
                    // Agregar variables
                    data.variables.forEach(variable => {
                        const option = document.createElement('option');
                        option.value = variable.Field;
                        option.textContent = variable.Field + (variable.Comment ? ` - ${variable.Comment}` : '');
                        select.appendChild(option);
                    });
                    
                    // Si hay una variable seleccionada previamente, marcarla
                    const seccion = seccionEditando;
                    if (seccion && seccion.contenido && seccion.contenido.variable_condicion) {
                        select.value = seccion.contenido.variable_condicion;
                    }
                } else {
                    console.error('Error al cargar variables:', data.message);
                }
            } catch (error) {
                console.error('Error al cargar variables:', error);
            }
        }

        // Función para actualizar la vista previa en línea
        function actualizarVistaPrevia() {
            const iframe = document.getElementById('preview-iframe');
            const loading = document.getElementById('preview-loading');
            const container = document.querySelector('.preview-container');
            
            if (iframe && loading && container) {
                // Mostrar loading
                loading.style.display = 'flex';
                container.classList.add('loading');
                
                // Actualizar el src del iframe
                iframe.src = iframe.src;
                
                // Ocultar loading cuando se cargue
                iframe.onload = function() {
                    setTimeout(() => {
                        loading.style.display = 'none';
                        container.classList.remove('loading');
                    }, 500);
                };
            }
        }

        // Función para manejar la carga de la vista previa
        function inicializarVistaPrevia() {
            const iframe = document.getElementById('preview-iframe');
            const loading = document.getElementById('preview-loading');
            const container = document.querySelector('.preview-container');
            
            if (iframe && loading && container) {
                // Mostrar loading inicial
                loading.style.display = 'flex';
                container.classList.add('loading');
                
                // Ocultar loading cuando se cargue
                iframe.onload = function() {
                    setTimeout(() => {
                        loading.style.display = 'none';
                        container.classList.remove('loading');
                    }, 1000);
                };
                
                // Manejar errores de carga
                iframe.onerror = function() {
                    loading.innerHTML = `
                        <div class="loading-spinner">
                            <i class="fas fa-exclamation-triangle" style="color: #f44336;"></i>
                            <p>Error al cargar la vista previa</p>
                            <button class="btn-action btn-secondary btn-sm" onclick="actualizarVistaPrevia()" style="margin-top: 1rem;">
                                <i class="fas fa-redo"></i> Reintentar
                            </button>
                        </div>
                    `;
                    container.classList.remove('loading');
                };
            }
        }

        // Función para ver una sección (solo lectura)
        function verSeccion(idSeccion) {
            const seccion = seccionesAgregadas.find(s => s.id == idSeccion);
            if (!seccion) {
                console.error('Sección no encontrada:', idSeccion);
                return;
            }

            seccionEditando = seccion;
            
            // Configurar el modal
            document.getElementById('modal-titulo').textContent = `Ver ${seccion.nombre}`;
            
            // Generar el contenido de solo lectura según el tipo de sección
            const modalBody = document.getElementById('modal-body');
            modalBody.innerHTML = generarContenidoSoloLectura(seccion);
            
            // Actualizar el footer para solo lectura
            const modalFooter = document.getElementById('modal-footer');
            modalFooter.innerHTML = `
                <button class="btn-action btn-secondary" onclick="cerrarModalEdicion()">
                    <i class="fas fa-times"></i>
                    Cerrar
                </button>
            `;
            
            // Mostrar el modal
            document.getElementById('modal-edicion').style.display = 'flex';
        }

        // Función para editar una sección
        function editarSeccion(idSeccion) {
            const seccion = seccionesAgregadas.find(s => s.id == idSeccion);
            if (!seccion) {
                console.error('Sección no encontrada:', idSeccion);
                return;
            }

            // Verificar si es de solo lectura
            if (seccion.solo_lectura) {
                verSeccion(idSeccion);
                return;
            }

            // Si es ruta-ejercitacion, abrir el editor de rutas
            if (seccion.tipo === 'ruta-ejercitacion') {
                abrirEditorRutas();
                return;
            }

            seccionEditando = seccion;
            
            // Configurar el modal
            document.getElementById('modal-titulo').textContent = `Editar ${seccion.nombre}`;
            
            // Generar el contenido del editor según el tipo de sección
            const modalBody = document.getElementById('modal-body');
            modalBody.innerHTML = generarFormularioEdicion(seccion);
            
            // Agregar event listeners después de que el modal esté renderizado
            if (seccion.tipo === 'archivos-pdf') {
                const pdfInput = document.getElementById('pdf-archivo');
                if (pdfInput) {
                    pdfInput.addEventListener('change', function() {
                        previsualizarPDF(this);
                    });
                    console.log('Event listener agregado para pdf-archivo');
                }
            } else if (seccion.tipo === 'seccion-video') {
                // Configurar cambio de tipo de video
                manejarCambioTipoVideo();
                
                // Agregar event listener para archivo de video
                const videoInput = document.getElementById('video-archivo');
                if (videoInput) {
                    videoInput.addEventListener('change', function() {
                        previsualizarVideo(this);
                    });
                    console.log('Event listener agregado para video-archivo');
                }
            } else if (seccion.tipo === 'seccion-podcast') {
                // Agregar event listener para archivo de podcast
                const podcastInput = document.getElementById('podcast-archivo');
                if (podcastInput) {
                    podcastInput.addEventListener('change', function() {
                        previsualizarAudio(this);
                    });
                    console.log('Event listener agregado para podcast-archivo');
                }
            } else if (seccion.tipo === 'respuesta-larga') {
                // Configurar checkboxes de respuesta larga
                manejarCheckboxRetroalimentacion();
                manejarCheckboxFechaLimite();
                console.log('Event listeners agregados para respuesta-larga');
            } else if (seccion.tipo === 'termino-actividad') {
                // Cargar variables de la base de datos
                cargarVariablesBD();
                console.log('Cargando variables de BD para termino-actividad');
            } else if (seccion.tipo === 'texto-rico') {
                // No necesitamos inicializar TinyMCE, es un textarea simple para HTML
                console.log('Editor de texto rico cargado (textarea HTML)');
            } else if (seccion.tipo === 'ruta-ejercitacion') {
                // Agregar event listener para archivo HTML
                const ejercitacionArchivoInput = document.getElementById('ejercitacion-archivo');
                if (ejercitacionArchivoInput) {
                    ejercitacionArchivoInput.addEventListener('change', function() {
                        previsualizarArchivoHTML(this);
                    });
                    console.log('Event listener agregado para ejercitacion-archivo');
                }
            } else if (seccion.tipo === 'geogebra-applet') {
                // Agregar event listener para URL de GeoGebra
                const geogebraUrlInput = document.getElementById('geogebra-url');
                if (geogebraUrlInput) {
                    geogebraUrlInput.addEventListener('input', function() {
                        previsualizarGeoGebraURL(this.value);
                    });
                    // Mostrar vista previa inicial si hay URL
                    if (geogebraUrlInput.value) {
                        previsualizarGeoGebraURL(geogebraUrlInput.value);
                    }
                    console.log('Event listener agregado para geogebra-url');
                }
            }
            
            // Actualizar el footer para edición
            const modalFooter = document.getElementById('modal-footer');
            modalFooter.innerHTML = `
                <button class="btn-action btn-secondary" onclick="cerrarModalEdicion()">
                    <i class="fas fa-times"></i>
                    Cancelar
                </button>
                <button class="btn-action btn-primary" onclick="guardarEdicionSeccion()">
                    <i class="fas fa-save"></i>
                    Guardar Cambios
                </button>
            `;
            
            // Mostrar el modal
            document.getElementById('modal-edicion').style.display = 'flex';
        }

        // Función para generar contenido de solo lectura
        function generarContenidoSoloLectura(seccion) {
            switch (seccion.tipo) {
                case 'aprendizaje-esperado':
                    return `
                        <div class="form-group">
                            <label>Título de la Sección</label>
                            <div class="readonly-field">${seccion.contenido?.titulo || 'Aprendizaje Esperado'}</div>
                        </div>
                        <div class="form-group">
                            <label>Contenido del Aprendizaje Esperado</label>
                            <div class="readonly-field">${seccion.contenido?.contenido || 'No hay contenido disponible'}</div>
                        </div>
                        <div class="form-group">
                            <label>Fuente</label>
                            <div class="readonly-field">
                                <i class="fas fa-database"></i> ${seccion.contenido?.fuente || 'configuracion_cursos.php'}
                            </div>
                        </div>
                    `;
                case 'criterios-evaluacion':
                    const criterios = seccion.contenido?.criterios || [];
                    return `
                        <div class="form-group">
                            <label>Título de la Sección</label>
                            <div class="readonly-field">${seccion.contenido?.titulo || 'Criterios de Evaluación'}</div>
                        </div>
                        <div class="form-group">
                            <label>Criterios de Evaluación</label>
                            <div class="criterios-readonly">
                                ${criterios.map(criterio => `
                                    <div class="criterio-readonly-item">
                                        <div class="criterio-codigo-readonly">${criterio.codigo}</div>
                                        <div class="criterio-descripcion-readonly">${criterio.descripcion}</div>
                                    </div>
                                `).join('')}
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Fuente</label>
                            <div class="readonly-field">
                                <i class="fas fa-database"></i> ${seccion.contenido?.fuente || 'configuracion_cursos.php'}
                            </div>
                        </div>
                    `;
                case 'contenidos':
                    const contenidos = seccion.contenido?.contenidos || [];
                    return `
                        <div class="form-group">
                            <label>Título de la Sección</label>
                            <div class="readonly-field">${seccion.contenido?.titulo || 'Contenidos'}</div>
                        </div>
                        <div class="form-group">
                            <label>Contenidos de la Unidad</label>
                            <div class="contenidos-readonly">
                                ${contenidos.map(contenido => `
                                    <div class="contenido-readonly-item">
                                        <div class="contenido-icono-readonly">
                                            <i class="${contenido.icono || 'fas fa-book'}"></i>
                                        </div>
                                        <div class="contenido-info-readonly">
                                            <div class="contenido-titulo-readonly">${contenido.titulo}</div>
                                            <div class="contenido-descripcion-readonly">${contenido.descripcion}</div>
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Fuente</label>
                            <div class="readonly-field">
                                <i class="fas fa-database"></i> ${seccion.contenido?.fuente || 'configuracion_cursos.php'}
                            </div>
                        </div>
                    `;
                default:
                    return `
                        <div class="form-group">
                            <label>Información de la Sección</label>
                            <div class="readonly-field">Esta sección es de solo lectura</div>
                        </div>
                    `;
            }
        }

        // Función para generar el formulario de edición según el tipo de sección
        function generarFormularioEdicion(seccion) {
            switch (seccion.tipo) {
                case 'titulo-1':
                    return `
                        <div class="form-group">
                            <label>Formato del Título</label>
                            <div class="info-box">
                                <strong>Actividad ${numeroActividad}:</strong> [Descripción que agregues aquí]
                            </div>
                        </div>
                        <div class="form-group">
                            <label for="titulo-descripcion">Descripción de la Actividad</label>
                            <input type="text" id="titulo-descripcion" value="${seccion.contenido?.descripcion || ''}" placeholder="Ej: Introducción a GeoGebra">
                        </div>
                    `;
                
                case 'aprendizaje-esperado':
                    return `
                        <div class="form-group">
                            <label for="aprendizaje-titulo">Título de la Sección</label>
                            <input type="text" id="aprendizaje-titulo" value="${seccion.contenido?.titulo || 'Aprendizaje Esperado'}" placeholder="Título de la sección">
                        </div>
                        <div class="form-group">
                            <label for="aprendizaje-contenido">Contenido del Aprendizaje Esperado</label>
                            <textarea id="aprendizaje-contenido" placeholder="Describe los objetivos de aprendizaje...">${seccion.contenido?.contenido || 'Reconocer y manejar la interfaz de la actividad, identificando sus principales elementos y funcionalidades.'}</textarea>
                        </div>
                    `;
                
                case 'criterios-evaluacion':
                    return `
                        <div class="form-group">
                            <label for="criterios-titulo">Título de la Sección</label>
                            <input type="text" id="criterios-titulo" value="${seccion.contenido?.titulo || 'Criterios de Evaluación'}" placeholder="Título de la sección">
                        </div>
                        <div class="form-group">
                            <label>Criterios de Evaluación</label>
                            <div id="criterios-lista">
                                ${generarListaCriterios(seccion.contenido?.criterios || [
                                    {codigo: '1.1.1', descripcion: 'Identificar correctamente los elementos principales de la actividad.'},
                                    {codigo: '1.1.2', descripcion: 'Completar la actividad siguiendo las instrucciones proporcionadas.'}
                                ])}
                            </div>
                            <button type="button" class="btn-action btn-secondary" onclick="agregarCriterio()" style="margin-top: 1rem;">
                                <i class="fas fa-plus"></i>
                                Agregar Criterio
                            </button>
                        </div>
                    `;
                
                case 'archivos-pdf':
                    return `
                        <div class="form-group">
                            <label for="pdf-titulo">Título del Documento</label>
                            <input type="text" id="pdf-titulo" value="${seccion.contenido?.titulo || 'Guía de la Actividad'}" placeholder="Título del documento">
                        </div>
                        <div class="form-group">
                            <label for="pdf-descripcion">Descripción del Documento</label>
                            <textarea id="pdf-descripcion" placeholder="Descripción del documento...">${seccion.contenido?.descripcion || 'Documento con las instrucciones y material de apoyo para la actividad.'}</textarea>
                        </div>
                        <div class="form-group">
                            <label for="pdf-archivo">Archivo PDF</label>
                            <input type="file" id="pdf-archivo" accept=".pdf" style="margin-top: 0.5rem;">
                            <div id="pdf-preview" style="margin-top: 1rem;"></div>
                            ${seccion.contenido?.archivo ? `<div class="archivo-actual">Archivo actual: ${seccion.contenido.archivo}</div>` : ''}
                        </div>
                    `;

                case 'seccion-video':
                    return `
                        <div class="form-group">
                            <label for="video-titulo">Título del Video</label>
                            <input type="text" id="video-titulo" value="${seccion.contenido?.titulo || 'Tutorial de la Actividad'}" placeholder="Título del video">
                        </div>
                        <div class="form-group">
                            <label for="video-descripcion">Descripción del Video</label>
                            <textarea id="video-descripcion" placeholder="Descripción del video...">${seccion.contenido?.descripcion || 'Video explicativo con los pasos para completar la actividad correctamente.'}</textarea>
                        </div>
                        <div class="form-group">
                            <label>Tipo de Video</label>
                            <div style="display: flex; gap: 1rem; margin: 0.5rem 0;">
                                <label style="display: flex; align-items: center; gap: 0.5rem;">
                                    <input type="radio" name="video-tipo" value="url" ${!seccion.contenido?.archivo ? 'checked' : ''}>
                                    <span>URL (YouTube, Vimeo, etc.)</span>
                                </label>
                                <label style="display: flex; align-items: center; gap: 0.5rem;">
                                    <input type="radio" name="video-tipo" value="archivo" ${seccion.contenido?.archivo ? 'checked' : ''}>
                                    <span>Archivo de Video</span>
                                </label>
                            </div>
                        </div>
                        <div class="form-group" id="video-url-group" style="display: ${!seccion.contenido?.archivo ? 'block' : 'none'};">
                            <label for="video-url">URL del Video</label>
                            <input type="url" id="video-url" value="${seccion.contenido?.url || ''}" placeholder="https://www.youtube.com/watch?v=...">
                            <small style="color: #666; font-size: 0.875rem;">Soporta YouTube, Vimeo y otros servicios.</small>
                            <div id="video-url-preview" style="margin-top: 1rem;"></div>
                        </div>
                        <div class="form-group" id="video-archivo-group" style="display: ${seccion.contenido?.archivo ? 'block' : 'none'};">
                            <label for="video-archivo">Archivo de Video</label>
                            <input type="file" id="video-archivo" accept=".mp4,.avi,.mov,.wmv,.webm" style="margin-top: 0.5rem;">
                            <div id="video-preview" style="margin-top: 1rem;"></div>
                            ${seccion.contenido?.archivo ? `<div class="archivo-actual">Archivo actual: ${seccion.contenido.archivo}</div>` : ''}
                        </div>
                        <div class="form-group">
                            <label for="video-duracion">Duración (opcional)</label>
                            <input type="text" id="video-duracion" value="${seccion.contenido?.duracion || ''}" placeholder="5:30 minutos">
                        </div>
                    `;

                case 'seccion-podcast':
                    return `
                        <div class="form-group">
                            <label for="podcast-titulo">Título del Podcast</label>
                            <input type="text" id="podcast-titulo" value="${seccion.contenido?.titulo || 'Audio Explicativo'}" placeholder="Título del podcast">
                        </div>
                        <div class="form-group">
                            <label for="podcast-descripcion">Descripción del Podcast</label>
                            <textarea id="podcast-descripcion" placeholder="Descripción del podcast...">${seccion.contenido?.descripcion || 'Audio con explicaciones detalladas sobre los conceptos de la actividad.'}</textarea>
                        </div>
                        <div class="form-group">
                            <label for="podcast-archivo">Archivo de Audio (MP3)</label>
                            <input type="file" id="podcast-archivo" accept=".mp3,.wav,.ogg" style="margin-top: 0.5rem;">
                            <div id="podcast-preview" style="margin-top: 1rem;"></div>
                            ${seccion.contenido?.archivo ? `<div class="archivo-actual">Archivo actual: ${seccion.contenido.archivo}</div>` : ''}
                        </div>
                        <div class="form-group">
                            <label for="podcast-duracion">Duración (opcional)</label>
                            <input type="text" id="podcast-duracion" value="${seccion.contenido?.duracion || ''}" placeholder="8:45 minutos">
                        </div>
                    `;

                case 'respuesta-larga':
                    const numeroRespuesta = seccion.numero_respuesta ? seccion.numero_respuesta.toString().padStart(2, '0') : '01';
                    return `
                        <div class="form-group">
                            <label for="respuesta-titulo-principal">Título Principal:</label>
                            <input type="text" id="respuesta-titulo-principal" class="form-control" value="${seccion.contenido?.titulo_principal || `Respuesta Larga ${numeroRespuesta}`}" placeholder="Ej: Respuesta Larga ${numeroRespuesta}">
                        </div>
                        <div class="form-group">
                            <label for="respuesta-subtitulo">Subtítulo:</label>
                            <input type="text" id="respuesta-subtitulo" class="form-control" value="${seccion.contenido?.subtitulo || 'Respuesta del Estudiante'}" placeholder="Ej: Respuesta del Estudiante">
                        </div>
                        <div class="form-group">
                            <label for="respuesta-descripcion">Descripción para el Estudiante:</label>
                            <textarea id="respuesta-descripcion" class="form-control" rows="3" placeholder="Ej: Escribe tu respuesta completa a la actividad...">${seccion.contenido?.descripcion || 'Escribe tu respuesta completa a la actividad. Asegúrate de incluir todos los puntos solicitados y fundamentar tus respuestas con ejemplos cuando sea necesario.'}</textarea>
                        </div>
                        <div class="form-group">
                            <label for="respuesta-min-palabras">Mínimo de Palabras:</label>
                            <input type="number" id="respuesta-min-palabras" class="form-control" value="${seccion.contenido?.min_palabras || 50}" min="1" placeholder="50">
                        </div>
                        <div class="form-group">
                            <label for="respuesta-max-palabras">Máximo de Palabras:</label>
                            <input type="number" id="respuesta-max-palabras" class="form-control" value="${seccion.contenido?.max_palabras || 100}" min="1" placeholder="100">
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="respuesta-incluir-fecha-limite" ${seccion.contenido?.incluir_fecha_limite ? 'checked' : ''}>
                                Establecer Fecha Límite para la Respuesta
                            </label>
                        </div>
                        <div class="form-group" id="fecha-limite-config" style="${seccion.contenido?.incluir_fecha_limite ? '' : 'display: none;'}">
                            <label for="respuesta-fecha-limite">Fecha Límite:</label>
                            <input type="datetime-local" id="respuesta-fecha-limite" class="form-control" value="${seccion.contenido?.fecha_limite || ''}">
                            <small class="form-text text-muted">Si no se establece, no habrá fecha límite</small>
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="respuesta-incluir-retro" ${seccion.contenido?.incluir_retroalimentacion ? 'checked' : ''}>
                                Incluir Sección de Retroalimentación Docente
                            </label>
                        </div>
                        <div class="form-group" id="retroalimentacion-config" style="${seccion.contenido?.incluir_retroalimentacion ? '' : 'display: none;'}">
                            <label for="respuesta-retro-descripcion">Descripción de Retroalimentación:</label>
                            <textarea id="respuesta-retro-descripcion" class="form-control" rows="2" placeholder="Ej: Proporciona retroalimentación constructiva...">${seccion.contenido?.retro_descripcion || 'Proporciona retroalimentación constructiva sobre la respuesta del estudiante. Incluye comentarios sobre fortalezas, áreas de mejora y sugerencias para el aprendizaje.'}</textarea>
                            <label style="margin-top: 1rem; display: block;">
                                <input type="checkbox" id="respuesta-incluir-calificacion" ${seccion.contenido?.incluir_calificacion ? 'checked' : ''}>
                                Incluir Campo de Calificación (0-100)
                            </label>
                        </div>
                    `;

                case 'termino-actividad':
                    return `
                        <div class="form-group">
                            <label for="termino-texto-boton">Texto del Botón:</label>
                            <input type="text" id="termino-texto-boton" class="form-control" value="${seccion.contenido?.texto_boton || 'Marcar como Completada'}" placeholder="Ej: Marcar como Completada">
                        </div>
                        <div class="form-group">
                            <label for="termino-variable-condicion">Variable de Condición:</label>
                            <select id="termino-variable-condicion" class="form-control">
                                <option value="">Seleccione una variable...</option>
                            </select>
                            <small class="form-text text-muted">El botón estará habilitado cuando esta variable sea igual a 1</small>
                        </div>
                        <div class="form-group">
                            <label for="termino-mensaje-exito">Mensaje de Éxito:</label>
                            <input type="text" id="termino-mensaje-exito" class="form-control" value="${seccion.contenido?.mensaje_exito || '¡Actividad completada exitosamente!'}" placeholder="Ej: ¡Actividad completada exitosamente!">
                        </div>
                    `;
                
                case 'texto-rico':
                    return `
                        <div class="form-group">
                            <label for="texto-rico-titulo">Título de la Sección:</label>
                            <input type="text" id="texto-rico-titulo" class="form-control" value="${seccion.contenido?.titulo || ''}" placeholder="Ej: Conceptos Clave">
                        </div>
                        <div class="form-group">
                            <label for="texto-rico-contenido">Contenido (HTML permitido):</label>
                            <textarea id="texto-rico-editor" class="form-control" rows="15" style="font-family: monospace; font-size: 13px;" placeholder="Escribe aquí el HTML... Puedes usar &lt;img src=&quot;url&quot;&gt; para imágenes y &lt;p&gt;...&lt;/p&gt; para párrafos">${seccion.contenido?.contenido || ''}</textarea>
                            <small class="form-text text-muted">
                                <strong>Ejemplos:</strong><br>
                                • Imagen: &lt;img src="ruta/imagen.jpg" alt="Descripción"&gt;<br>
                                • Párrafo: &lt;p&gt;Texto aquí&lt;/p&gt;<br>
                                • Títulos: &lt;h2&gt;Título&lt;/h2&gt;<br>
                                • Listas: &lt;ul&gt;&lt;li&gt;Item&lt;/li&gt;&lt;/ul&gt;
                            </small>
                        </div>
                    `;
                
                                case 'ruta-ejercitacion':
                    return `
                        <div class="form-group">
                            <label for="ejercitacion-titulo">Título de la Sección:</label>
                            <input type="text" id="ejercitacion-titulo" class="form-control" value="${seccion.contenido?.titulo || ''}" placeholder="Ej: Ruta de Ejercicios">
                        </div>
                        <div class="form-group">
                            <label for="ejercitacion-archivo">Archivo HTML con Ejercicios:</label>
                            <input type="file" id="ejercitacion-archivo" accept=".html" style="margin-top: 0.5rem;">
                            <div id="ejercitacion-preview" style="margin-top: 1rem;"></div>
                            ${seccion.contenido?.archivo ? `<div class="archivo-actual">Archivo actual: ${seccion.contenido.archivo}</div>` : ''}
                            <small class="form-text text-muted">
                                Carga un archivo HTML que contenga los ejercicios interactivos.
                            </small>
                        </div>
                    `;
                
                case 'geogebra-applet':
                    return `
                        <div class="form-group">
                            <label for="geogebra-titulo">Título de la Sección:</label>
                            <input type="text" id="geogebra-titulo" class="form-control" value="${seccion.contenido?.titulo || ''}" placeholder="Ej: Actividad con GeoGebra">
                        </div>
                        <div class="form-group">
                            <label for="geogebra-url">Código de Incrustación de GeoGebra:</label>
                            <textarea id="geogebra-url" class="form-control" rows="3" placeholder='<iframe src="https://www.geogebra.org/material/iframe/id/..." width="..." height="..."></iframe>'>${seccion.contenido?.url || ''}</textarea>
                            <small class="form-text text-muted">
                                Pega aquí el código de incrustación completo del iframe de GeoGebra.
                            </small>
                            <div id="geogebra-preview" style="margin-top: 1rem;"></div>
                        </div>
                    `;
                 
                default:
                    return `
                        <div class="form-group">
                            <label for="seccion-titulo">Título de la Sección</label>
                            <input type="text" id="seccion-titulo" value="${seccion.contenido?.titulo || seccion.nombre}" placeholder="Título de la sección">
                        </div>
                        <div class="form-group">
                            <label for="seccion-contenido">Contenido</label>
                            <textarea id="seccion-contenido" placeholder="Contenido de la sección...">${seccion.contenido?.contenido || ''}</textarea>
                        </div>
                    `;
            }
        }

        // Función para generar la lista de criterios
        function generarListaCriterios(criterios) {
            return criterios.map((criterio, index) => `
                <div class="criterio-editor" style="display: flex; gap: 1rem; margin-bottom: 1rem; align-items: center;">
                    <input type="text" value="${criterio.codigo}" placeholder="Código" style="width: 100px; flex-shrink: 0;" class="criterio-codigo">
                    <input type="text" value="${criterio.descripcion}" placeholder="Descripción del criterio" style="flex: 1;" class="criterio-descripcion">
                    <button type="button" onclick="eliminarCriterio(this)" style="background: #f44336; border: none; color: white; padding: 0.5rem; border-radius: 4px; cursor: pointer;">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `).join('');
        }

        // Función para agregar un criterio nuevo
        function agregarCriterio() {
            const lista = document.getElementById('criterios-lista');
            const nuevoCriterio = document.createElement('div');
            nuevoCriterio.className = 'criterio-editor';
            nuevoCriterio.style.cssText = 'display: flex; gap: 1rem; margin-bottom: 1rem; align-items: center;';
            nuevoCriterio.innerHTML = `
                <input type="text" placeholder="Código" style="width: 100px; flex-shrink: 0;" class="criterio-codigo">
                <input type="text" placeholder="Descripción del criterio" style="flex: 1;" class="criterio-descripcion">
                <button type="button" onclick="eliminarCriterio(this)" style="background: #f44336; border: none; color: white; padding: 0.5rem; border-radius: 4px; cursor: pointer;">
                    <i class="fas fa-times"></i>
                </button>
            `;
            lista.appendChild(nuevoCriterio);
        }

        // Función para eliminar un criterio
        function eliminarCriterio(boton) {
            boton.parentElement.remove();
        }

        // Función para previsualizar PDF
        function previsualizarPDF(input) {
            console.log('previsualizarPDF llamada', input);
            const preview = document.getElementById('pdf-preview');
            console.log('Elemento preview encontrado:', preview);
            
            if (!preview) {
                console.error('No se encontró el elemento pdf-preview');
                return;
            }
            
            if (input.files && input.files[0]) {
                const file = input.files[0];
                console.log('Archivo seleccionado:', file.name, file.type, file.size);
                
                if (file.type === 'application/pdf') {
                    // Crear URL temporal para el archivo
                    const fileURL = URL.createObjectURL(file);
                    
                    preview.innerHTML = `
                        <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; border: 1px solid #dee2e6; margin-top: 1rem; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
                            <!-- Información del archivo -->
                            <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem; padding: 1rem; background: linear-gradient(135deg, #eff6ff, #dbeafe); border-radius: 8px; border: 1px solid rgba(30, 64, 175, 0.1);">
                                <div style="font-size: 2rem; color: #1e40af;">
                                    <i class="fas fa-file-pdf"></i>
                                </div>
                                <div>
                                    <h4 style="color: #1e40af; font-size: 1.1rem; font-weight: 600; margin: 0 0 0.25rem 0;">${file.name}</h4>
                                    <p style="color: #64748b; font-size: 0.875rem; margin: 0;">
                                        Tamaño: ${(file.size / 1024 / 1024).toFixed(2)} MB
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Vista previa del PDF con estilos mejorados -->
                            <div style="border: 1px solid #ddd; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);">
                                <iframe 
                                    src="${fileURL}#toolbar=1&navpanes=1&scrollbar=1" 
                                    width="100%" 
                                    height="400" 
                                    style="border: none; display: block;"
                                    title="Vista previa del PDF">
                                </iframe>
                            </div>
                            
                            <!-- Mensaje informativo sin botones -->
                            <div style="margin-top: 1rem; text-align: center; padding: 0.75rem; background: #f0f9ff; border-radius: 6px; border: 1px solid #bae6fd;">
                                <p style="margin: 0; color: #0369a1; font-size: 0.875rem; font-weight: 500;">
                                    <i class="fas fa-info-circle" style="margin-right: 0.5rem;"></i>
                                    Vista previa del archivo seleccionado. Guarda la sección para finalizar.
                                </p>
                            </div>
                        </div>
                    `;
                    console.log('Vista previa PDF con iframe mejorada mostrada correctamente');
                } else {
                    preview.innerHTML = `
                        <div style="background: #f8d7da; color: #721c24; padding: 1rem; border-radius: 8px; border: 1px solid #f5c6cb; margin-top: 1rem;">
                            <i class="fas fa-exclamation-triangle" style="margin-right: 0.5rem;"></i>
                            Por favor selecciona un archivo PDF válido.
                        </div>
                    `;
                    input.value = '';
                    console.log('Archivo no es PDF, limpiado');
                }
            } else {
                console.log('No hay archivos seleccionados');
                preview.innerHTML = '';
            }
        }
        
        // Función para descargar el PDF de la vista previa
        function descargarPDF(url, filename) {
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            console.log('Descarga iniciada:', filename);
        }

        // Función para manejar cambio de tipo de video
        function manejarCambioTipoVideo() {
            const radios = document.querySelectorAll('input[name="video-tipo"]');
            const urlGroup = document.getElementById('video-url-group');
            const archivoGroup = document.getElementById('video-archivo-group');
            
            radios.forEach(radio => {
                radio.addEventListener('change', function() {
                    if (this.value === 'url') {
                        urlGroup.style.display = 'block';
                        archivoGroup.style.display = 'none';
                    } else {
                        urlGroup.style.display = 'none';
                        archivoGroup.style.display = 'block';
                    }
                });
            });
            
            // Agregar event listener para el input de URL
            const urlInput = document.getElementById('video-url');
            if (urlInput) {
                urlInput.addEventListener('input', function() {
                    previsualizarVideoURL(this.value);
                });
                // Mostrar vista previa inicial si hay URL
                if (urlInput.value) {
                    previsualizarVideoURL(urlInput.value);
                }
            }
        }

        // Función para manejar checkbox de retroalimentación
        function manejarCheckboxRetroalimentacion() {
            const checkbox = document.getElementById('respuesta-incluir-retro');
            const configDiv = document.getElementById('retroalimentacion-config');
            
            if (checkbox && configDiv) {
                checkbox.addEventListener('change', function() {
                    if (this.checked) {
                        configDiv.style.display = 'block';
                    } else {
                        configDiv.style.display = 'none';
                    }
                });
            }
        }

        // Función para manejar checkbox de fecha límite
        function manejarCheckboxFechaLimite() {
            const checkbox = document.getElementById('respuesta-incluir-fecha-limite');
            const configDiv = document.getElementById('fecha-limite-config');
            
            if (checkbox && configDiv) {
                checkbox.addEventListener('change', function() {
                    if (this.checked) {
                        configDiv.style.display = 'block';
                        // Establecer fecha por defecto si no hay una
                        const fechaInput = document.getElementById('respuesta-fecha-limite');
                        if (fechaInput && !fechaInput.value) {
                            const ahora = new Date();
                            ahora.setDate(ahora.getDate() + 7); // 7 días desde hoy
                            fechaInput.value = ahora.toISOString().slice(0, 16);
                        }
                    } else {
                        configDiv.style.display = 'none';
                    }
                });
            }
        }

        // Función para previsualizar video desde URL
        function previsualizarVideoURL(url) {
            const preview = document.getElementById('video-url-preview');
            
            if (!preview) {
                console.error('No se encontró el elemento video-url-preview');
                return;
            }
            
            if (url && url.trim()) {
                // Convertir URL a formato embed
                let embedUrl = url;
                if (url.includes('youtube.com/watch')) {
                    embedUrl = url.replace('youtube.com/watch?v=', 'youtube.com/embed/');
                } else if (url.includes('youtu.be/')) {
                    embedUrl = url.replace('youtu.be/', 'youtube.com/embed/');
                } else if (url.includes('youtube.com/live/')) {
                    embedUrl = url.replace('youtube.com/live/', 'youtube.com/embed/');
                } else if (url.includes('vimeo.com/')) {
                    const videoId = url.split('vimeo.com/')[1];
                    embedUrl = `https://player.vimeo.com/video/${videoId}`;
                }
                
                preview.innerHTML = `
                    <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; border: 1px solid #dee2e6; margin-top: 1rem; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
                        <!-- Información de la URL -->
                        <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem; padding: 1rem; background: linear-gradient(135deg, #eff6ff, #dbeafe); border-radius: 8px; border: 1px solid rgba(30, 64, 175, 0.1);">
                            <div style="font-size: 2rem; color: #1e40af;">
                                <i class="fas fa-play-circle"></i>
                            </div>
                            <div>
                                <h4 style="color: #1e40af; font-size: 1.1rem; font-weight: 600; margin: 0 0 0.25rem 0;">Video desde URL</h4>
                                <p style="color: #64748b; font-size: 0.875rem; margin: 0; word-break: break-all;">
                                    ${url}
                                </p>
                            </div>
                        </div>
                        
                        <!-- Vista previa del video -->
                        <div style="border: 1px solid #ddd; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);">
                            <iframe 
                                src="${embedUrl}" 
                                width="100%" 
                                height="300" 
                                style="border: none; display: block;"
                                title="Vista previa del video"
                                onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                            </iframe>
                            <div style="display: none; padding: 40px; text-align: center; background: #f8f9fa;">
                                <i class="fas fa-exclamation-triangle" style="font-size: 2rem; color: #dc3545; margin-bottom: 1rem;"></i>
                                <p style="color: #6c757d; margin: 0 0 1rem 0;">El video no se puede mostrar en vista previa</p>
                                <a href="${url}" target="_blank" style="background: #007bff; color: white; padding: 0.5rem 1rem; border-radius: 4px; text-decoration: none; display: inline-block;">
                                    <i class="fas fa-external-link-alt"></i> Abrir en YouTube
                                </a>
                            </div>
                        </div>
                        
                        <!-- Mensaje informativo -->
                        <div style="margin-top: 1rem; text-align: center; padding: 0.75rem; background: #f0f9ff; border-radius: 6px; border: 1px solid #bae6fd;">
                            <p style="margin: 0; color: #0369a1; font-size: 0.875rem; font-weight: 500;">
                                <i class="fas fa-info-circle" style="margin-right: 0.5rem;"></i>
                                Vista previa del video desde URL. Guarda la sección para finalizar.
                            </p>
                        </div>
                    </div>
                `;
                console.log('Vista previa de video URL mostrada correctamente');
            } else {
                preview.innerHTML = '';
            }
        }

        // Función para previsualizar video
        function previsualizarVideo(input) {
            console.log('previsualizarVideo llamada', input);
            const preview = document.getElementById('video-preview');
            
            if (!preview) {
                console.error('No se encontró el elemento video-preview');
                return;
            }
            
            if (input.files && input.files[0]) {
                const file = input.files[0];
                console.log('Archivo de video seleccionado:', file.name, file.type, file.size);
                
                if (file.type.startsWith('video/')) {
                    const fileURL = URL.createObjectURL(file);
                    
                    preview.innerHTML = `
                        <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; border: 1px solid #dee2e6; margin-top: 1rem; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
                            <!-- Información del archivo -->
                            <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem; padding: 1rem; background: linear-gradient(135deg, #eff6ff, #dbeafe); border-radius: 8px; border: 1px solid rgba(30, 64, 175, 0.1);">
                                <div style="font-size: 2rem; color: #1e40af;">
                                    <i class="fas fa-video"></i>
                                </div>
                                <div>
                                    <h4 style="color: #1e40af; font-size: 1.1rem; font-weight: 600; margin: 0 0 0.25rem 0;">${file.name}</h4>
                                    <p style="color: #64748b; font-size: 0.875rem; margin: 0;">
                                        Tamaño: ${(file.size / 1024 / 1024).toFixed(2)} MB
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Vista previa del video -->
                            <div style="border: 1px solid #ddd; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);">
                                <video 
                                    src="${fileURL}" 
                                    controls 
                                    style="width: 100%; height: 300px; display: block;"
                                    title="Vista previa del video">
                                </video>
                            </div>
                            
                            <!-- Mensaje informativo -->
                            <div style="margin-top: 1rem; text-align: center; padding: 0.75rem; background: #f0f9ff; border-radius: 6px; border: 1px solid #bae6fd;">
                                <p style="margin: 0; color: #0369a1; font-size: 0.875rem; font-weight: 500;">
                                    <i class="fas fa-info-circle" style="margin-right: 0.5rem;"></i>
                                    Vista previa del video seleccionado. Guarda la sección para finalizar.
                                </p>
                            </div>
                        </div>
                    `;
                    console.log('Vista previa de video mostrada correctamente');
                } else {
                    preview.innerHTML = `
                        <div style="background: #f8d7da; color: #721c24; padding: 1rem; border-radius: 8px; border: 1px solid #f5c6cb; margin-top: 1rem;">
                            <i class="fas fa-exclamation-triangle" style="margin-right: 0.5rem;"></i>
                            Por favor selecciona un archivo de video válido.
                        </div>
                    `;
                    input.value = '';
                }
            } else {
                preview.innerHTML = '';
            }
        }

        // Función para previsualizar audio
        function previsualizarAudio(input) {
            console.log('previsualizarAudio llamada', input);
            const preview = document.getElementById('podcast-preview');
            
            if (!preview) {
                console.error('No se encontró el elemento podcast-preview');
                return;
            }
            
            if (input.files && input.files[0]) {
                const file = input.files[0];
                console.log('Archivo de audio seleccionado:', file.name, file.type, file.size);
                
                if (file.type.startsWith('audio/')) {
                    const fileURL = URL.createObjectURL(file);
                    
                    preview.innerHTML = `
                        <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; border: 1px solid #dee2e6; margin-top: 1rem; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
                            <!-- Información del archivo -->
                            <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem; padding: 1rem; background: linear-gradient(135deg, #eff6ff, #dbeafe); border-radius: 8px; border: 1px solid rgba(30, 64, 175, 0.1);">
                                <div style="font-size: 2rem; color: #1e40af;">
                                    <i class="fas fa-headphones"></i>
                                </div>
                                <div>
                                    <h4 style="color: #1e40af; font-size: 1.1rem; font-weight: 600; margin: 0 0 0.25rem 0;">${file.name}</h4>
                                    <p style="color: #64748b; font-size: 0.875rem; margin: 0;">
                                        Tamaño: ${(file.size / 1024 / 1024).toFixed(2)} MB
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Reproductor de audio -->
                            <div style="border: 1px solid #ddd; border-radius: 8px; padding: 1rem; background: white; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);">
                                <audio 
                                    src="${fileURL}" 
                                    controls 
                                    style="width: 100%; display: block;"
                                    title="Vista previa del audio">
                                </audio>
                            </div>
                            
                            <!-- Mensaje informativo -->
                            <div style="margin-top: 1rem; text-align: center; padding: 0.75rem; background: #f0f9ff; border-radius: 6px; border: 1px solid #bae6fd;">
                                <p style="margin: 0; color: #0369a1; font-size: 0.875rem; font-weight: 500;">
                                    <i class="fas fa-info-circle" style="margin-right: 0.5rem;"></i>
                                    Vista previa del audio seleccionado. Guarda la sección para finalizar.
                                </p>
                            </div>
                        </div>
                    `;
                    console.log('Vista previa de audio mostrada correctamente');
                } else {
                    preview.innerHTML = `
                        <div style="background: #f8d7da; color: #721c24; padding: 1rem; border-radius: 8px; border: 1px solid #f5c6cb; margin-top: 1rem;">
                            <i class="fas fa-exclamation-triangle" style="margin-right: 0.5rem;"></i>
                            Por favor selecciona un archivo de audio válido.
                        </div>
                    `;
                    input.value = '';
                }
            } else {
                preview.innerHTML = '';
            }
        }

        // Función para previsualizar archivo HTML
        function previsualizarArchivoHTML(input) {
            console.log('previsualizarArchivoHTML llamada', input);
            const preview = document.getElementById('ejercitacion-preview');
            
            if (!preview) {
                console.error('No se encontró el elemento ejercitacion-preview');
                return;
            }
            
            if (input.files && input.files[0]) {
                const file = input.files[0];
                console.log('Archivo HTML seleccionado:', file.name, file.type, file.size);
                
                if (file.type === 'text/html' || file.name.endsWith('.html')) {
                    preview.innerHTML = `
                        <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; border: 1px solid #dee2e6; margin-top: 1rem; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
                            <!-- Información del archivo -->
                            <div style="display: flex; align-items: center; gap: 1rem; padding: 1rem; background: linear-gradient(135deg, #eff6ff, #dbeafe); border-radius: 8px; border: 1px solid rgba(30, 64, 175, 0.1);">
                                <div style="font-size: 2rem; color: #1e40af;">
                                    <i class="fas fa-file-code"></i>
                                </div>
                                <div>
                                    <h4 style="color: #1e40af; font-size: 1.1rem; font-weight: 600; margin: 0 0 0.25rem 0;">${file.name}</h4>
                                    <p style="color: #64748b; font-size: 0.875rem; margin: 0;">
                                        Tamaño: ${(file.size / 1024).toFixed(2)} KB
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Mensaje informativo -->
                            <div style="margin-top: 1rem; text-align: center; padding: 0.75rem; background: #f0f9ff; border-radius: 6px; border: 1px solid #bae6fd;">
                                <p style="margin: 0; color: #0369a1; font-size: 0.875rem; font-weight: 500;">
                                    <i class="fas fa-info-circle" style="margin-right: 0.5rem;"></i>
                                    Archivo HTML cargado. Guarda la sección para finalizar.
                                </p>
                            </div>
                        </div>
                    `;
                    console.log('Vista previa de archivo HTML mostrada correctamente');
                } else {
                    preview.innerHTML = `
                        <div style="background: #f8d7da; color: #721c24; padding: 1rem; border-radius: 8px; border: 1px solid #f5c6cb; margin-top: 1rem;">
                            <i class="fas fa-exclamation-triangle" style="margin-right: 0.5rem;"></i>
                            Por favor selecciona un archivo HTML válido (.html).
                        </div>
                    `;
                    input.value = '';
                }
            } else {
                preview.innerHTML = '';
            }
        }

        // Función para previsualizar URL de GeoGebra
        function previsualizarGeoGebraURL(iframeCode) {
            const preview = document.getElementById('geogebra-preview');
            
            if (!preview) {
                console.error('No se encontró el elemento geogebra-preview');
                return;
            }
            
            if (iframeCode && iframeCode.trim()) {
                // Extraer el src del iframe
                let embedUrl = '';
                let hasValidIframe = false;
                
                // Buscar el atributo src en el código del iframe
                const srcMatch = iframeCode.match(/src="([^"]+)"/);
                if (srcMatch && srcMatch[1]) {
                    embedUrl = srcMatch[1];
                    hasValidIframe = true;
                }
                
                if (hasValidIframe && embedUrl.includes('geogebra.org')) {
                    preview.innerHTML = `
                        <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; border: 1px solid #dee2e6; margin-top: 1rem; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
                            <!-- Información de la URL -->
                            <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem; padding: 1rem; background: linear-gradient(135deg, #fff3cd, #ffeeba); border-radius: 8px; border: 1px solid rgba(255, 193, 7, 0.3);">
                                <div style="font-weight: 700; font-size: 1.2rem; color: #ff6b00;">
                                    GG
                                </div>
                                <div>
                                    <h4 style="color: #856404; font-size: 1.1rem; font-weight: 600; margin: 0 0 0.25rem 0;">GeoGebra Applet</h4>
                                    <p style="color: #856404; font-size: 0.875rem; margin: 0; word-break: break-all;">
                                        ${embedUrl}
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Vista previa del applet -->
                            <div style="border: 1px solid #ddd; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);">
                                <iframe 
                                    src="${embedUrl}" 
                                    width="100%" 
                                    height="400" 
                                    style="border: none; display: block;"
                                    title="Vista previa del applet GeoGebra">
                                </iframe>
                            </div>
                            
                            <!-- Mensaje informativo -->
                            <div style="margin-top: 1rem; text-align: center; padding: 0.75rem; background: #fff3cd; border-radius: 6px; border: 1px solid #ffc107;">
                                <p style="margin: 0; color: #856404; font-size: 0.875rem; font-weight: 500;">
                                    <i class="fas fa-info-circle" style="margin-right: 0.5rem;"></i>
                                    Vista previa del applet GeoGebra. Guarda la sección para finalizar.
                                </p>
                            </div>
                        </div>
                    `;
                    console.log('Vista previa de GeoGebra mostrada correctamente');
                } else {
                    preview.innerHTML = `
                        <div style="margin-top: 1rem; padding: 1rem; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 6px; color: #721c24;">
                            <i class="fas fa-exclamation-triangle" style="margin-right: 0.5rem;"></i>
                            El código del iframe no es válido. Por favor, pega el código completo del iframe de GeoGebra.
                        </div>
                    `;
                }
            } else {
                preview.innerHTML = '';
            }
        }

        // Función para subir archivos multimedia (PDF, Video, Audio, HTML)
        async function subirArchivo(archivo, tipoSeccion) {
            const formData = new FormData();
            formData.append('archivo', archivo);
            formData.append('tipo_seccion', tipoSeccion);
            formData.append('curso', parametrosActuales.curso);
            formData.append('unidad', parametrosActuales.unidad);
            formData.append('actividad', parametrosActuales.actividad);

            const response = await fetch('subir_archivo.php', {
                method: 'POST',
                body: formData
            });

            return await response.json();
        }

        // Función para cerrar el modal de edición
        function cerrarModalEdicion() {
            document.getElementById('modal-edicion').style.display = 'none';
            seccionEditando = null;
        }

        // Función para guardar la edición de la sección
        async function guardarEdicionSeccion() {
            if (!seccionEditando) {
                console.error('No hay sección editando');
                return;
            }

            // Recopilar los datos del formulario según el tipo de sección
            const nuevosDatos = recopilarDatosFormulario(seccionEditando.tipo);
            
            // Si hay un archivo, subirlo primero según el tipo de sección
            if (nuevosDatos.archivo && ['archivos-pdf', 'seccion-video', 'seccion-podcast', 'ruta-ejercitacion'].includes(seccionEditando.tipo)) {
                try {
                    const resultado = await subirArchivo(nuevosDatos.archivo, seccionEditando.tipo);
                    if (resultado.success) {
                        nuevosDatos.archivo = resultado.nombreArchivo;
                        nuevosDatos.ruta = resultado.ruta;
                    } else {
                        alert('Error al subir el archivo: ' + resultado.message);
                        return;
                    }
                } catch (error) {
                    console.error('Error al subir archivo:', error);
                    alert('Error al subir el archivo');
                    return;
                }
            }
            
            // Actualizar la sección
            if (!seccionEditando.contenido) {
                seccionEditando.contenido = {};
            }
            
            Object.assign(seccionEditando.contenido, nuevosDatos);
            
            // Actualizar la lista visual
            actualizarListaSecciones();
            
            // Cerrar el modal
            cerrarModalEdicion();
            
            // Mostrar mensaje de éxito
            mostrarMensajeExito('Sección actualizada exitosamente');
            
            console.log('Sección actualizada:', seccionEditando);
        }

        // Función para recopilar los datos del formulario
        function recopilarDatosFormulario(tipoSeccion) {
            const datos = {};
            
            switch (tipoSeccion) {
                case 'titulo-1':
                    datos.descripcion = document.getElementById('titulo-descripcion')?.value || '';
                    break;
                
                case 'aprendizaje-esperado':
                    datos.titulo = document.getElementById('aprendizaje-titulo')?.value || '';
                    datos.contenido = document.getElementById('aprendizaje-contenido')?.value || '';
                    break;
                
                case 'criterios-evaluacion':
                    datos.titulo = document.getElementById('criterios-titulo')?.value || '';
                    datos.criterios = [];
                    
                    const criteriosEditor = document.querySelectorAll('.criterio-editor');
                    criteriosEditor.forEach(criterio => {
                        const codigo = criterio.querySelector('.criterio-codigo')?.value;
                        const descripcion = criterio.querySelector('.criterio-descripcion')?.value;
                        
                        if (codigo && descripcion) {
                            datos.criterios.push({ codigo, descripcion });
                        }
                    });
                    break;
                
                case 'archivos-pdf':
                    datos.titulo = document.getElementById('pdf-titulo')?.value || '';
                    datos.descripcion = document.getElementById('pdf-descripcion')?.value || '';
                    // El archivo se manejará por separado en el servidor
                    const pdfArchivoInput = document.getElementById('pdf-archivo');
                    if (pdfArchivoInput && pdfArchivoInput.files.length > 0) {
                        datos.archivo = pdfArchivoInput.files[0];
                    }
                    break;

                case 'seccion-video':
                    datos.titulo = document.getElementById('video-titulo')?.value || '';
                    datos.descripcion = document.getElementById('video-descripcion')?.value || '';
                    datos.duracion = document.getElementById('video-duracion')?.value || '';
                    
                    // Determinar si es URL o archivo
                    const tipoVideo = document.querySelector('input[name="video-tipo"]:checked')?.value;
                    if (tipoVideo === 'url') {
                        datos.url = document.getElementById('video-url')?.value || '';
                        datos.tipo_contenido = 'url';
                    } else {
                        const videoInput = document.getElementById('video-archivo');
                        if (videoInput && videoInput.files.length > 0) {
                            datos.archivo = videoInput.files[0];
                            datos.tipo_contenido = 'archivo';
                        }
                    }
                    break;

                case 'seccion-podcast':
                    datos.titulo = document.getElementById('podcast-titulo')?.value || '';
                    datos.descripcion = document.getElementById('podcast-descripcion')?.value || '';
                    datos.duracion = document.getElementById('podcast-duracion')?.value || '';
                    
                    const podcastInput = document.getElementById('podcast-archivo');
                    if (podcastInput && podcastInput.files.length > 0) {
                        datos.archivo = podcastInput.files[0];
                    }
                    break;

                case 'respuesta-larga':
                    datos.titulo_principal = document.getElementById('respuesta-titulo-principal')?.value || 'Respuesta Larga 01';
                    datos.subtitulo = document.getElementById('respuesta-subtitulo')?.value || 'Respuesta del Estudiante';
                    datos.descripcion = document.getElementById('respuesta-descripcion')?.value || '';
                    datos.min_palabras = parseInt(document.getElementById('respuesta-min-palabras')?.value) || 50;
                    datos.max_palabras = parseInt(document.getElementById('respuesta-max-palabras')?.value) || 100;
                    datos.incluir_fecha_limite = document.getElementById('respuesta-incluir-fecha-limite')?.checked || false;
                    datos.fecha_limite = document.getElementById('respuesta-fecha-limite')?.value || '';
                    datos.incluir_retroalimentacion = document.getElementById('respuesta-incluir-retro')?.checked || false;
                    datos.retro_descripcion = document.getElementById('respuesta-retro-descripcion')?.value || '';
                    datos.incluir_calificacion = document.getElementById('respuesta-incluir-calificacion')?.checked || false;
                    break;
                
                case 'termino-actividad':
                    datos.texto_boton = document.getElementById('termino-texto-boton')?.value || 'Terminar Actividad 01';
                    datos.variable_condicion = document.getElementById('termino-variable-condicion')?.value || '';
                    datos.mensaje_exito = document.getElementById('termino-mensaje-exito')?.value || '¡Actividad completada exitosamente!';
                    break;
                
                case 'texto-rico':
                    datos.titulo = document.getElementById('texto-rico-titulo')?.value || '';
                    // Obtener el contenido HTML del textarea
                    datos.contenido = document.getElementById('texto-rico-editor')?.value || '';
                    break;
                
                case 'ruta-ejercitacion':
                    datos.titulo = document.getElementById('ejercitacion-titulo')?.value || '';
                    const ejercitacionInput = document.getElementById('ejercitacion-archivo');
                    if (ejercitacionInput && ejercitacionInput.files.length > 0) {
                        datos.archivo = ejercitacionInput.files[0];
                    }
                    break;
                
                case 'geogebra-applet':
                    datos.titulo = document.getElementById('geogebra-titulo')?.value || '';
                    // Guardar el código completo del iframe
                    const geogebraTextarea = document.getElementById('geogebra-url');
                    datos.url = geogebraTextarea ? geogebraTextarea.value.trim() : '';
                    break;
                
                default:
                    datos.titulo = document.getElementById('seccion-titulo')?.value || '';
                    datos.contenido = document.getElementById('seccion-contenido')?.value || '';
                    break;
            }
            
            return datos;
        }

        // Función para generar HTML de una sección (versión simplificada para vista previa)
        function generarHTMLSeccion(seccion, numero) {
            let html = `<!-- Sección ${numero}: ${seccion.nombre} -->\n`;
            html += `<section class="${seccion.tipo}">\n`;
            
            switch (seccion.tipo) {
                case 'titulo-1':
                    const descripcionTitulo = seccion.contenido?.descripcion || '';
                    html += `    <h1 class="titulo-1">Actividad ${numeroActividad}${descripcionTitulo ? ': ' + descripcionTitulo : ''}</h1>\n`;
                    break;
                case 'aprendizaje-esperado':
                    html += `    <h3>Aprendizaje Esperado</h3>\n`;
                    html += `    <p>${seccion.contenido?.contenido || seccion.contenido?.aprendizaje_esperado || 'Reconocer y manejar la interfaz de la actividad, identificando sus principales elementos y funcionalidades.'}</p>\n`;
                    break;
                case 'criterios-evaluacion':
                    html += `    <h3>Criterios de Evaluación</h3>\n`;
                    const criterios = seccion.contenido?.criterios || [];
                    criterios.forEach(criterio => {
                        html += `    <div class="criterio-item">\n`;
                        html += `        <div class="criterio-codigo">Criterio ${criterio.codigo || '1.1.1'}</div>\n`;
                        html += `        <div class="criterio-descripcion">${criterio.descripcion || 'Descripción del criterio'}</div>\n`;
                        html += `    </div>\n`;
                    });
                    break;
                case 'contenidos':
                    html += `    <h3>Contenidos</h3>\n`;
                    const contenidos = seccion.contenido?.contenidos || [];
                    contenidos.forEach(contenido => {
                        html += `    <div class="contenido-item">\n`;
                        html += `        <div class="contenido-header">\n`;
                        html += `            <div class="contenido-icono"><i class="${contenido.icono || 'fas fa-book'}"></i></div>\n`;
                        html += `            <h4 class="contenido-titulo">${contenido.titulo || 'Contenido'}</h4>\n`;
                        html += `        </div>\n`;
                        html += `        <p class="contenido-descripcion">${contenido.descripcion || 'Descripción del contenido'}</p>\n`;
                        html += `    </div>\n`;
                    });
                    break;
                case 'archivos-pdf':
                    const tituloPDF = seccion.contenido?.titulo || 'Guía de la Actividad';
                    const descripcionPDF = seccion.contenido?.descripcion || 'Documento con las instrucciones y material de apoyo para la actividad.';
                    html += `    <h3>Documento PDF</h3>\n`;
                    html += `    <div class="pdf-item">\n`;
                    html += `        <div class="pdf-header">\n`;
                    html += `            <div class="pdf-icono"><i class="fas fa-file-pdf"></i></div>\n`;
                    html += `            <h4 class="pdf-titulo">${tituloPDF}</h4>\n`;
                    html += `        </div>\n`;
                    html += `        <p class="pdf-descripcion">${descripcionPDF}</p>\n`;
                    if (seccion.contenido?.archivo) {
                        html += `        <div class="pdf-descarga">\n`;
                        html += `            <a href="documentos/${seccion.contenido.archivo}" target="_blank" class="btn-descarga-pdf">\n`;
                        html += `                <i class="fas fa-download"></i> Descargar PDF\n`;
                        html += `            </a>\n`;
                        html += `        </div>\n`;
                    }
                    html += `    </div>\n`;
                    break;
                case 'seccion-video':
                    html += `    <h3>Video Educativo</h3>\n`;
                    html += `    <div class="video-container">\n`;
                    html += `        <div class="video-titulo">Tutorial de la Actividad</div>\n`;
                    html += `        <div class="video-descripcion">Video explicativo con los pasos para completar la actividad correctamente.</div>\n`;
                    html += `        <div class="video-embed">\n`;
                    html += `            <iframe src="https://www.youtube.com/embed/dQw4w9WgXcQ" title="Tutorial de la Actividad" allowfullscreen></iframe>\n`;
                    html += `        </div>\n`;
                    html += `    </div>\n`;
                    break;
                case 'seccion-podcast':
                    html += `    <h3>Podcast Educativo</h3>\n`;
                    html += `    <div class="podcast-container">\n`;
                    html += `        <div class="podcast-titulo">Audio Explicativo</div>\n`;
                    html += `        <div class="podcast-descripcion">Audio con explicaciones detalladas sobre los conceptos de la actividad.</div>\n`;
                    html += `    </div>\n`;
                    break;
                case 'seccion-cargar-archivo':
                    html += `    <h3>Cargar Archivo</h3>\n`;
                    html += `    <div class="cargar-archivo-container">\n`;
                    html += `        <div class="cargar-archivo-titulo">Subir Documento de Trabajo</div>\n`;
                    html += `        <div class="cargar-archivo-descripcion">Sube tu documento de trabajo en formato PDF o Word.</div>\n`;
                    html += `        <div class="upload-area">\n`;
                    html += `            <div class="upload-icon">📁</div>\n`;
                    html += `            <div class="upload-text">Arrastra y suelta tu archivo aquí</div>\n`;
                    html += `        </div>\n`;
                    html += `    </div>\n`;
                    break;
                case 'respuesta-larga':
                    const numeroRespuestaHTML = seccion.numero_respuesta ? seccion.numero_respuesta.toString().padStart(2, '0') : '01';
                    const tituloPrincipal = seccion.contenido?.titulo_principal || `Respuesta Larga ${numeroRespuestaHTML}`;
                    const subtitulo = seccion.contenido?.subtitulo || 'Respuesta del Estudiante';
                    html += `    <h3>${tituloPrincipal}</h3>\n`;
                    html += `    <div class="espacio-respuesta">\n`;
                    html += `        <h4>${subtitulo}</h4>\n`;
                    html += `        <textarea class="textarea-respuesta" placeholder="Escribe tu respuesta aquí..."></textarea>\n`;
                    html += `    </div>\n`;
                    break;
                case 'seccion-foro':
                    html += `    <h3>Foro de Opinión</h3>\n`;
                    html += `    <div class="foro-container">\n`;
                    html += `        <div class="foro-titulo">¿Qué opinas sobre esta actividad?</div>\n`;
                    html += `        <div class="foro-descripcion">Comparte tu experiencia y opinión sobre esta actividad.</div>\n`;
                    html += `    </div>\n`;
                    break;
                case 'termino-actividad':
                    const textoBoton = seccion.contenido?.texto_boton || 'Marcar como Completada';
                    const variableCondicion = seccion.contenido?.variable_condicion || '';
                    const mensajeExito = seccion.contenido?.mensaje_exito || '¡Actividad completada exitosamente!';
                    const numeroActividadTermino = numeroActividad || 1;
                    
                    html += `    <div class="termino-actividad-container">\n`;
                    html += `        <button class="btn-terminar-actividad" id="btn-terminar-act-${numeroActividadTermino}" onclick="marcarActividadComoCompletada(${numeroActividadTermino}, '${variableCondicion}')" disabled>\n`;
                    html += `            <i class="fas fa-check-circle"></i>\n`;
                    html += `            ${textoBoton}\n`;
                    html += `        </button>\n`;
                    html += `        <div id="mensaje-exito-${numeroActividadTermino}" style="display: none; margin-top: 1rem; padding: 1rem; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 8px; color: #155724;">\n`;
                    html += `            <i class="fas fa-check-circle"></i> ${mensajeExito}\n`;
                    html += `        </div>\n`;
                    html += `    </div>\n`;
                    break;
                default:
                    html += `    <p>Contenido de la sección ${seccion.nombre}</p>\n`;
                    break;
            }
            
            html += `</section>\n\n`;
            return html;
        }

        // Mostrar información de debug en consola
        if (parametrosActuales.archivoExiste) {
            console.log('✅ Archivo detectado:', parametrosActuales.nombreArchivo);
        } else {
            console.log('❌ Archivo no encontrado:', parametrosActuales.nombreArchivo);
        }

        // Inicializar la lista de secciones
        document.addEventListener('DOMContentLoaded', function() {
            console.log('🔍 DEBUG - DOMContentLoaded');
            console.log('Secciones detectadas:', seccionesDetectadas);
            console.log('Cantidad:', seccionesDetectadas ? seccionesDetectadas.length : 0);
            
            // Si hay secciones detectadas, cargarlas automáticamente
            if (seccionesDetectadas && seccionesDetectadas.length > 0) {
                console.log('✅ Secciones detectadas automáticamente:', seccionesDetectadas);
                seccionesAgregadas = [...seccionesDetectadas];
                console.log('📋 Secciones agregadas al array:', seccionesAgregadas);
                
                // Reasignar números de respuesta larga si hay alguna
                reasignarNumerosRespuestaLarga();
                
                actualizarListaSecciones();
                console.log('🔄 Lista actualizada. Cantidad en array:', seccionesAgregadas.length);
                
                // Mostrar mensaje informativo
                mostrarMensajeDetectado(seccionesDetectadas.length);
            } else {
                console.log('❌ No hay secciones detectadas');
                actualizarListaSecciones();
            }
            
            // Inicializar vista previa si existe
            if (parametrosActuales.archivoExiste) {
                inicializarVistaPrevia();
            }
        });

        // Función para mostrar mensaje cuando se detectan secciones
        function mostrarMensajeDetectado(cantidad) {
            const mensaje = document.createElement('div');
            mensaje.className = 'mensaje-detectado';
            mensaje.innerHTML = `
                <div style="background: linear-gradient(135deg, #4caf50 0%, #45a049 100%); 
                           color: white; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;
                           display: flex; align-items: center; gap: 0.5rem;">
                    <i class="fas fa-magic"></i>
                    <span>Se detectaron automáticamente ${cantidad} sección(es) en el archivo existente</span>
                </div>
            `;
            
            const contenedor = document.querySelector('.detector-container');
            contenedor.insertBefore(mensaje, contenedor.firstChild);
            
            // Remover el mensaje después de 5 segundos
            setTimeout(() => {
                if (mensaje.parentNode) {
                    mensaje.parentNode.removeChild(mensaje);
                }
            }, 5000);
        }
    </script>
</body>
</html>

<?php
// Función para detectar secciones en un archivo existente respetando el orden
function detectarSeccionesEnArchivo($contenido) {
    $secciones_detectadas = [];
    
    // Patrón general para detectar todas las secciones en orden
    $patron_general = '/<section class="([^"]+)">/';
    
    // Buscar todas las secciones en el orden que aparecen
    if (preg_match_all($patron_general, $contenido, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE)) {
        // Ordenar por posición en el archivo
        usort($matches, function($a, $b) {
            return $a[0][1] - $b[0][1];
        });
        
        // Procesar cada sección encontrada
        foreach ($matches as $match) {
            $tipo = $match[1][0];
            
            // Verificar que sea un tipo válido
            if (esTipoSeccionValido($tipo)) {
                $seccion = [
                    'id' => time() + rand(1000, 9999) + count($secciones_detectadas), // ID numérico único
                    'tipo' => $tipo,
                    'nombre' => obtenerNombreSeccion($tipo),
                    'descripcion' => obtenerDescripcionSeccion($tipo),
                    'icono' => obtenerIconoSeccion($tipo)
                ];
                
                // Si es aprendizaje esperado, cargar contenido desde configuración
                if ($tipo === 'aprendizaje-esperado') {
                    $seccion['contenido'] = obtenerAprendizajeEsperadoConfiguracion($curso, $unidad);
                    $seccion['solo_lectura'] = true; // Marcar como solo lectura
                }
                
                // Si es criterios de evaluación, cargar contenido desde configuración
                if ($tipo === 'criterios-evaluacion') {
                    $seccion['contenido'] = obtenerCriteriosEvaluacionConfiguracion($curso, $unidad);
                    $seccion['solo_lectura'] = true; // Marcar como solo lectura
                }
                
                // Si es contenidos, cargar contenido desde configuración
                if ($tipo === 'contenidos') {
                    $seccion['contenido'] = obtenerContenidosConfiguracion($curso, $unidad);
                    $seccion['solo_lectura'] = true; // Marcar como solo lectura
                }
                
                // Si es título, extraer la descripción del archivo
                if ($tipo === 'titulo-1') {
                    $seccion['contenido'] = extraerContenidoTitulo($contenido, $match[0][1]);
                }
                
                // Si es archivos-pdf, extraer los datos del documento
                if ($tipo === 'archivos-pdf') {
                    $seccion['contenido'] = extraerContenidoPDF($contenido, $match[0][1]);
                }
                
                // Si es seccion-video, extraer los datos del video
                if ($tipo === 'seccion-video') {
                    $seccion['contenido'] = extraerContenidoVideo($contenido, $match[0][1]);
                }
                
                // Si es seccion-podcast, extraer los datos del podcast
                if ($tipo === 'seccion-podcast') {
                    $seccion['contenido'] = extraerContenidoPodcast($contenido, $match[0][1]);
                }
                
                // Si es respuesta-larga, extraer los datos de la respuesta larga
                if ($tipo === 'respuesta-larga') {
                    $seccion['contenido'] = extraerContenidoRespuestaLarga($contenido, $match[0][1]);
                    // El número se asignará automáticamente por JavaScript
                }
                
                // Si es termino-actividad, extraer los datos del término de actividad
                if ($tipo === 'termino-actividad') {
                    $seccion['contenido'] = extraerContenidoTerminoActividad($contenido, $match[0][1]);
                }
                
                // Si es texto-rico, extraer el contenido HTML
                if ($tipo === 'texto-rico') {
                    $seccion['contenido'] = extraerContenidoTextoRico($contenido, $match[0][1]);
                }
                
                // Si es ruta-ejercitacion, extraer el contenido del archivo HTML
                if ($tipo === 'ruta-ejercitacion') {
                    $seccion['contenido'] = extraerContenidoRutaEjercitacion($contenido, $match[0][1]);
                }
                
                // Si es geogebra-applet, extraer el contenido del applet
                if ($tipo === 'geogebra-applet') {
                    $seccion['contenido'] = extraerContenidoGeoGebra($contenido, $match[0][1]);
                }
                
                $secciones_detectadas[] = $seccion;
            }
        }
    }
    
    return $secciones_detectadas;
}

// Función auxiliar para verificar si un tipo de sección es válido
function esTipoSeccionValido($tipo) {
    $tipos_validos = [
        'titulo-1',
        'aprendizaje-esperado',
        'criterios-evaluacion',
        'contenidos',
        'archivos-pdf',
        'seccion-video',
        'seccion-podcast',
        'seccion-cargar-archivo',
        'respuesta-larga',
        'seccion-foro',
        'termino-actividad',
        'texto-rico',
        'ruta-ejercitacion',
        'geogebra-applet'
    ];
    
    return in_array($tipo, $tipos_validos);
}

// Función para normalizar el formato de unidad
function normalizarUnidad($unidad) {
    // Convertir "Unidad 01" a "Unidad 1", "Unidad 02" a "Unidad 2", etc.
    return preg_replace('/Unidad 0*(\d+)/', 'Unidad $1', $unidad);
}

// Función para extraer el contenido del título desde un archivo existente
function extraerContenidoTitulo($contenido, $posicion_inicio) {
    // Buscar el contenido del título después de la etiqueta <section>
    $posicion_h1 = strpos($contenido, '<h1 class="titulo-1">', $posicion_inicio);
    if ($posicion_h1 !== false) {
        $posicion_inicio_h1 = $posicion_h1 + strlen('<h1 class="titulo-1">');
        $posicion_fin_h1 = strpos($contenido, '</h1>', $posicion_inicio_h1);
        
        if ($posicion_fin_h1 !== false) {
            $texto_titulo = trim(substr($contenido, $posicion_inicio_h1, $posicion_fin_h1 - $posicion_inicio_h1));
            
            // Extraer la descripción después de "Actividad X: "
            if (preg_match('/^Actividad \d+: (.+)$/', $texto_titulo, $matches)) {
                return ['descripcion' => $matches[1]];
            } elseif (preg_match('/^Actividad \d+$/', $texto_titulo)) {
                return ['descripcion' => ''];
            }
        }
    }
    
    return ['descripcion' => ''];
}

function extraerContenidoPDF($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección PDF después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo' => 'Guía de la Actividad',
        'descripcion' => 'Documento con las instrucciones y material de apoyo para la actividad.',
        'archivo' => null
    ];
    
    // Extraer el título del PDF
    if (preg_match('/<h4 class="pdf-titulo">(.+?)<\/h4>/', $substring, $matches)) {
        $datos['titulo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la descripción del PDF
    if (preg_match('/<p class="pdf-descripcion">(.+?)<\/p>/s', $substring, $matches)) {
        $datos['descripcion'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el nombre del archivo
    if (preg_match('/<div class="pdf-preview-filename">(.+?)<\/div>/', $substring, $matches)) {
        $datos['archivo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    } elseif (preg_match('/href="documentos\/(.+?)"/', $substring, $matches)) {
        $datos['archivo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    return $datos;
}

// Función para extraer el contenido del video desde un archivo existente
function extraerContenidoVideo($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección video después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo' => 'Tutorial de la Actividad',
        'descripcion' => 'Video explicativo con los pasos para completar la actividad correctamente.',
        'duracion' => '',
        'url' => '',
        'archivo' => null,
        'tipo_contenido' => 'url'
    ];
    
    // Extraer el título del video
    if (preg_match('/<div class="video-titulo">(.+?)<\/div>/', $substring, $matches)) {
        $datos['titulo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la descripción del video
    if (preg_match('/<div class="video-descripcion">\s*(.+?)\s*<\/div>/s', $substring, $matches)) {
        $datos['descripcion'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la duración del video
    if (preg_match('/<span class="video-duracion">⏱️ (.+?)<\/span>/', $substring, $matches)) {
        $datos['duracion'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Detectar si es URL (iframe) o archivo (video)
    if (preg_match('/<iframe[^>]+src="([^"]+)"/', $substring, $matches)) {
        $url = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
        
        // Convertir URL de embed de vuelta a URL normal si es YouTube
        if (strpos($url, 'youtube.com/embed/') !== false) {
            // Extraer el ID del video
            $videoId = str_replace('https://www.youtube.com/embed/', '', $url);
            $videoId = strtok($videoId, '?'); // Remover parámetros adicionales
            
            // Intentar determinar si es live o video normal
            // Para simplificar, asumimos que si viene de embed, lo convertimos a watch
            $url = "https://www.youtube.com/watch?v=" . $videoId;
        }
        
        $datos['url'] = $url;
        $datos['tipo_contenido'] = 'url';
    } elseif (preg_match('/<video[^>]+src="videos\/(.+?)"/', $substring, $matches)) {
        $datos['archivo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
        $datos['tipo_contenido'] = 'archivo';
    }
    
    return $datos;
}

// Función para extraer el contenido del podcast desde un archivo existente
function extraerContenidoPodcast($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección podcast después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo' => 'Audio Explicativo',
        'descripcion' => 'Audio con explicaciones detalladas sobre los conceptos de la actividad.',
        'duracion' => '',
        'archivo' => null
    ];
    
    // Extraer el título del podcast
    if (preg_match('/<div class="podcast-titulo">(.+?)<\/div>/', $substring, $matches)) {
        $datos['titulo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la descripción del podcast
    if (preg_match('/<div class="podcast-descripcion">\s*(.+?)\s*<\/div>/s', $substring, $matches)) {
        $datos['descripcion'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la duración del podcast
    if (preg_match('/<span class="podcast-duracion">⏱️ (.+?)<\/span>/', $substring, $matches)) {
        $datos['duracion'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el nombre del archivo de audio
    if (preg_match('/<audio[^>]+src="podcasts\/(.+?)"/', $substring, $matches)) {
        $datos['archivo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    return $datos;
}

function extraerContenidoRespuestaLarga($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección respuesta larga después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo_principal' => 'Respuesta Larga 01',
        'subtitulo' => 'Respuesta del Estudiante',
        'descripcion' => 'Escribe tu respuesta completa a la actividad. Asegúrate de incluir todos los puntos solicitados y fundamentar tus respuestas con ejemplos cuando sea necesario.',
        'min_palabras' => 50,
        'max_palabras' => 100,
        'incluir_retroalimentacion' => false,
        'retro_descripcion' => 'Proporciona retroalimentación constructiva sobre la respuesta del estudiante. Incluye comentarios sobre fortalezas, áreas de mejora y sugerencias para el aprendizaje.',
        'incluir_calificacion' => false
    ];
    
    // Extraer el título principal (h3)
    if (preg_match('/<h3>(.+?)<\/h3>/', $substring, $matches)) {
        $datos['titulo_principal'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el subtítulo (h4)
    if (preg_match('/<h4>(.+?)<\/h4>/', $substring, $matches)) {
        $datos['subtitulo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la descripción
    if (preg_match('/<div class="espacio-respuesta-descripcion">\s*(.+?)\s*<\/div>/s', $substring, $matches)) {
        $datos['descripcion'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer mínimo y máximo de palabras del placeholder del textarea
    if (preg_match('/placeholder="[^"]*mínimo (\d+) palabras[^"]*máximo (\d+) palabras[^"]*"/', $substring, $matches)) {
        $datos['min_palabras'] = intval($matches[1]);
        $datos['max_palabras'] = intval($matches[2]);
    }
    
    // Verificar si existe sección de retroalimentación
    if (preg_match('/<div class="retroalimentacion-mejorada">/', $substring)) {
        $datos['incluir_retroalimentacion'] = true;
        
        // Extraer descripción de retroalimentación
        if (preg_match('/<div class="espacio-respuesta-descripcion">\s*(.+?)\s*<\/div>/s', $substring, $matches, PREG_OFFSET_CAPTURE, strpos($substring, 'retroalimentacion-mejorada'))) {
            $datos['retro_descripcion'] = html_entity_decode(trim(strip_tags($matches[1][0])), ENT_QUOTES, 'UTF-8');
        }
        
        // Verificar si incluye calificación
        if (preg_match('/<input[^>]*id="calificacion-[^"]*"[^>]*>/', $substring)) {
            $datos['incluir_calificacion'] = true;
        }
    }

    return $datos;
}

function extraerContenidoTerminoActividad($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección término de actividad después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'texto_boton' => 'Marcar como Completada',
        'variable_condicion' => '',
        'mensaje_exito' => '¡Actividad completada exitosamente!'
    ];
    
    // Extraer el texto del botón
    if (preg_match('/<button[^>]*class="btn-terminar-actividad"[^>]*>\s*<i[^>]*><\/i>\s*(.+?)\s*<\/button>/s', $substring, $matches)) {
        $datos['texto_boton'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer la variable de condición del onclick
    if (preg_match('/onclick="marcarActividadComoCompletada\([^,]+, \'([^\']+)\'\)"/', $substring, $matches)) {
        $datos['variable_condicion'] = html_entity_decode($matches[1], ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el mensaje de éxito
    if (preg_match('/<div id="mensaje-exito-\d+"[^>]*>\s*<i[^>]*><\/i>\s*(.+?)\s*<\/div>/s', $substring, $matches)) {
        $datos['mensaje_exito'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    return $datos;
}

function extraerContenidoTextoRico($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección texto rico después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo' => '',
        'contenido' => ''
    ];
    
    // Extraer el título (opcional)
    if (preg_match('/<h3 class="texto-rico-titulo">(.+?)<\/h3>/s', $substring, $matches)) {
        $datos['titulo'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el contenido HTML
    if (preg_match('/<div class="texto-rico-contenido">(.*?)<\/div>/s', $substring, $matches)) {
        $datos['contenido'] = trim($matches[1]);
    }
    
    return $datos;
}

function extraerContenidoRutaEjercitacion($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección ruta de ejercitación después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo' => '',
        'archivo' => null
    ];
    
    // Extraer el título (opcional)
    if (preg_match('/<h3 class="ejercitacion-titulo">(.+?)<\/h3>/s', $substring, $matches)) {
        $datos['titulo'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el archivo HTML (del src del iframe o del atributo data-archivo)
    if (preg_match('/<iframe[^>]+src="rutas\/(.+?)"/', $substring, $matches)) {
        $datos['archivo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    } elseif (preg_match('/<div[^>]+data-archivo="(.+?)"/', $substring, $matches)) {
        $datos['archivo'] = html_entity_decode(trim($matches[1]), ENT_QUOTES, 'UTF-8');
    } else {
        // Si no se encuentra un archivo específico, construir el nombre automáticamente
        global $curso, $unidad, $actividad;
        
        // Normalizar formato de unidad
        $unidad_normalizada = '';
        if (preg_match('/Unidad (\d+)/', $unidad, $matches)) {
            $unidad_normalizada = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
        } else {
            $unidad_normalizada = str_pad($unidad, 2, '0', STR_PAD_LEFT);
        }
        
        $actividad_padded = str_pad($actividad, 2, '0', STR_PAD_LEFT);
        $nombre_archivo = "ruta_{$curso}_u{$unidad_normalizada}_act{$actividad_padded}.php";
        
        // Verificar si el archivo existe
        $ruta_archivo = "../rutas/ruta_{$curso}_u{$unidad_normalizada}_act{$actividad_padded}/{$nombre_archivo}";
        if (file_exists($ruta_archivo)) {
            $datos['archivo'] = "ruta_{$curso}_u{$unidad_normalizada}_act{$actividad_padded}/{$nombre_archivo}";
        }
    }
    
    return $datos;
}

function extraerContenidoGeoGebra($contenido, $posicion_inicio) {
    // Buscar el contenido de la sección GeoGebra después de la etiqueta <section>
    $substring = substr($contenido, $posicion_inicio);
    
    $datos = [
        'titulo' => '',
        'url' => ''
    ];
    
    // Extraer el título (opcional)
    if (preg_match('/<h3 class="geogebra-titulo">(.+?)<\/h3>/s', $substring, $matches)) {
        $datos['titulo'] = html_entity_decode(trim(strip_tags($matches[1])), ENT_QUOTES, 'UTF-8');
    }
    
    // Extraer el código completo del iframe
    if (preg_match('/<div class="geogebra-iframe-container">(.+?)<\/div>/s', $substring, $container_match)) {
        // Limpiar el código del iframe (remover espacios extras pero mantener el formato)
        $iframe_code = trim($container_match[1]);
        
        // Si contiene un iframe de GeoGebra, guardar el código completo
        if (strpos($iframe_code, 'geogebra.org') !== false) {
            $datos['url'] = $iframe_code;
        }
    }
    
    return $datos;
}

// Función para obtener el aprendizaje esperado desde la configuración
function obtenerAprendizajeEsperadoConfiguracion($curso = null, $unidad = null) {
    global $cursos;
    
    // Usar parámetros pasados o los de la URL
    $curso = $curso ?: (isset($_GET['curso']) ? $_GET['curso'] : '');
    $unidad = $unidad ?: (isset($_GET['unidad']) ? $_GET['unidad'] : '');
    
    // Normalizar formato de unidad
    $unidad_normalizada = normalizarUnidad($unidad);
    
    if ($curso && $unidad_normalizada && isset($cursos[$curso]['unidades'][$unidad_normalizada])) {
        $aprendizaje_esperado = $cursos[$curso]['unidades'][$unidad_normalizada]['aprendizaje_esperado'];
        return [
            'titulo' => 'Aprendizaje Esperado',
            'contenido' => $aprendizaje_esperado,
            'fuente' => 'configuracion_cursos.php'
        ];
    }
    
    // Fallback si no se encuentra en configuración
    return [
        'titulo' => 'Aprendizaje Esperado',
        'contenido' => 'Reconocer y manejar la interfaz de la actividad, identificando sus principales elementos y funcionalidades.',
        'fuente' => 'default'
    ];
}

// Función para obtener los criterios de evaluación desde la configuración
function obtenerCriteriosEvaluacionConfiguracion($curso = null, $unidad = null) {
    global $cursos;
    
    // Usar parámetros pasados o los de la URL
    $curso = $curso ?: (isset($_GET['curso']) ? $_GET['curso'] : '');
    $unidad = $unidad ?: (isset($_GET['unidad']) ? $_GET['unidad'] : '');
    
    // Normalizar formato de unidad
    $unidad_normalizada = normalizarUnidad($unidad);
    
    if ($curso && $unidad_normalizada && isset($cursos[$curso]['unidades'][$unidad_normalizada])) {
        $criterios_config = $cursos[$curso]['unidades'][$unidad_normalizada]['criterios_evaluacion'];
        
        // Convertir array asociativo a array de objetos con código y descripción
        $criterios = [];
        foreach ($criterios_config as $codigo => $descripcion) {
            $criterios[] = [
                'codigo' => $codigo,
                'descripcion' => $descripcion
            ];
        }
        
        return [
            'titulo' => 'Criterios de Evaluación',
            'criterios' => $criterios,
            'fuente' => 'configuracion_cursos.php'
        ];
    }
    
    // Fallback si no se encuentra en configuración
    return [
        'titulo' => 'Criterios de Evaluación',
        'criterios' => [
            [
                'codigo' => '1.1.1',
                'descripcion' => 'Identificar correctamente los elementos principales de la actividad.'
            ],
            [
                'codigo' => '1.1.2',
                'descripcion' => 'Completar la actividad siguiendo las instrucciones proporcionadas.'
            ]
        ],
        'fuente' => 'default'
    ];
}

// Función para obtener los contenidos desde la configuración
function obtenerContenidosConfiguracion($curso = null, $unidad = null) {
    global $cursos;
    
    // Usar parámetros pasados o los de la URL
    $curso = $curso ?: (isset($_GET['curso']) ? $_GET['curso'] : '');
    $unidad = $unidad ?: (isset($_GET['unidad']) ? $_GET['unidad'] : '');
    
    // Normalizar formato de unidad
    $unidad_normalizada = normalizarUnidad($unidad);
    
    if ($curso && $unidad_normalizada && isset($cursos[$curso]['unidades'][$unidad_normalizada])) {
        $contenidos_config = $cursos[$curso]['unidades'][$unidad_normalizada]['contenidos'];
        
        // Los contenidos ya están en el formato correcto
        return [
            'titulo' => 'Contenidos',
            'contenidos' => $contenidos_config,
            'fuente' => 'configuracion_cursos.php'
        ];
    }
    
    // Fallback si no se encuentra en configuración
    return [
        'titulo' => 'Contenidos',
        'contenidos' => [
            [
                'titulo' => 'Conceptos básicos',
                'descripcion' => 'Introducción a los conceptos fundamentales de la actividad.',
                'icono' => 'fas fa-book'
            ],
            [
                'titulo' => 'Aplicación práctica',
                'descripcion' => 'Ejercicios y actividades para aplicar los conceptos aprendidos.',
                'icono' => 'fas fa-tasks'
            ]
        ],
        'fuente' => 'default'
    ];
}

// Función para obtener el nombre de una sección
function obtenerNombreSeccion($tipo) {
    $nombres = [
        'titulo-1' => 'Título de la Actividad',
        'aprendizaje-esperado' => 'Aprendizaje Esperado',
        'criterios-evaluacion' => 'Criterios de Evaluación',
        'contenidos' => 'Contenidos',
        'archivos-pdf' => 'Documento PDF',
        'seccion-video' => 'Video Educativo',
        'seccion-podcast' => 'Podcast Educativo',
        'seccion-cargar-archivo' => 'Cargar Archivo',
        'respuesta-larga' => 'Respuesta Larga',
        'seccion-foro' => 'Foro de Opinión',
        'termino-actividad' => 'Término de Actividad',
        'texto-rico' => 'Texto con Fórmulas',
        'ruta-ejercitacion' => 'Ruta de Ejercitación Progresiva',
        'geogebra-applet' => 'Applet GeoGebra'
    ];
    
    return isset($nombres[$tipo]) ? $nombres[$tipo] : 'Sección Desconocida';
}

// Función para obtener la descripción de una sección
function obtenerDescripcionSeccion($tipo) {
    $descripciones = [
        'titulo-1' => 'Encabezado principal de la actividad',
        'aprendizaje-esperado' => 'Objetivos de aprendizaje de la actividad',
        'criterios-evaluacion' => 'Criterios para evaluar el desempeño',
        'contenidos' => 'Lista de contenidos con iconos',
        'archivos-pdf' => 'Vista previa y descarga de PDF',
        'seccion-video' => 'Reproductor de video YouTube/Vimeo',
        'seccion-podcast' => 'Reproductor de audio personalizado',
        'seccion-cargar-archivo' => 'Subida de documentos con drag & drop',
        'respuesta-larga' => 'Área de texto para respuesta del estudiante',
        'seccion-foro' => 'Discusión y comentarios de estudiantes',
        'termino-actividad' => 'Botón para marcar actividad como completada',
        'texto-rico' => 'Editor de texto con fórmulas matemáticas e imágenes',
        'ruta-ejercitacion' => 'Ejercicios interactivos progresivos desde archivo HTML',
        'geogebra-applet' => 'Applet interactivo de GeoGebra incrustado en la actividad'
    ];
    
    return isset($descripciones[$tipo]) ? $descripciones[$tipo] : 'Descripción no disponible';
}

// Función para obtener el ícono de una sección
function obtenerIconoSeccion($tipo) {
    $iconos = [
        'titulo-1' => 'fas fa-heading',
        'aprendizaje-esperado' => 'fas fa-bullseye',
        'criterios-evaluacion' => 'fas fa-clipboard-check',
        'contenidos' => 'fas fa-list-ul',
        'archivos-pdf' => 'fas fa-file-pdf',
        'seccion-video' => 'fas fa-play-circle',
        'seccion-podcast' => 'fas fa-microphone',
        'seccion-cargar-archivo' => 'fas fa-upload',
        'respuesta-larga' => 'fas fa-edit',
        'seccion-foro' => 'fas fa-comments',
        'termino-actividad' => 'fas fa-check-circle',
        'texto-rico' => 'fas fa-square-root-alt',
        'ruta-ejercitacion' => 'fas fa-route',
        'geogebra-applet' => 'GG'
    ];
    
    return isset($iconos[$tipo]) ? $iconos[$tipo] : 'fas fa-question';
}
?>